{-------------------------------------------------------------------------------

The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is "UnitMain.pas" released at June 12nd, 2007.

The Initial Developer of the Original Code is

  Priyatna
  Website: http://www.priyatna.org
  Email: me@priyatna.org
  Copyright (c) 2007-2008 Priyatna
  All Rights Reserved.

Contributor(s):
  - Alexander Herz <aherz.kazan@arcor.de>, search feature and improvements.
  - Ludovic Leffet <ludovic.leffet@wanadoo.fr> bugs correction and minor improvements.
  - Mickal Vanneufville <mickael.vanneufville@gmail.com> bugs fix and improvements.

Description: Ant Movie Catalog Viewer main form.

Known Issues: -

Last Modified: 2012-05-27 by Mickal Vanneufville

-------------------------------------------------------------------------------}

unit UnitMain;

interface

{$WARN SYMBOL_PLATFORM OFF}

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, ExtCtrls, OleCtrls, SHDocVw, ImgList, MovieClass, Fields,
  StdCtrls, NiceGallery, NiceSideBar, Menus, Buttons, UContainer,
  ButtonComps, NameSpace, ShellAPI, PngImage, GroupSortUtils, Math;

const
  ttLow           = -1;
  ttNone          = -1;
  ttOriginal      =  0;
  ttTranslated    =  1;
  ttMedia         =  2;
  ttHigh          =  2;
type
  TTitleType = ttLow..ttHigh;
  TFormMain = class(TForm)
    ImageList1: TImageList;
    Panel1: TPanel;
    WebBrowser1: TWebBrowser;
    Label1: TLabel;
    Label2: TLabel;
    NiceGallery1: TNiceGallery;
    Splitter1: TSplitter;
    PopupMenu1: TPopupMenu;
    Category1: TMenuItem;
    Rating1: TMenuItem;
    Year1: TMenuItem;
    Producer1: TMenuItem;
    Director1: TMenuItem;
    Country1: TMenuItem;
    MovieNumber1: TMenuItem;
    Panel2: TPanel;
    NiceSideBar1: TNiceSideBar;
    Panel3: TPanel;
    LabelGroup: TLabel;
    Decade1: TMenuItem;
    Panel4: TPanel;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    BtnOpen: TEncartaButton;
    BtnEdit: TEncartaButton;
    BtnSettings: TEncartaButton;
    BtnCateg: TImageButton;
    BtnPlay: TEncartaButton;
    Groupby1: TMenuItem;
    None1: TMenuItem;
    N1: TMenuItem;
    Ascending1: TMenuItem;
    Descending1: TMenuItem;
    OriginalTitle1: TMenuItem;
    TranslatedTitle1: TMenuItem;
    Actors1: TMenuItem;
    Languages1: TMenuItem;
    Media1: TMenuItem;
    MediaType1: TMenuItem;
    DateAdded1: TMenuItem;
    Borrower1: TMenuItem;
    Length1: TMenuItem;
    VideoFormat1: TMenuItem;
    VideoBitrate1: TMenuItem;
    AudioFormat1: TMenuItem;
    AudioBitrate1: TMenuItem;
    Resolution1: TMenuItem;
    Framerate1: TMenuItem;
    Subtitles1: TMenuItem;
    Disks1: TMenuItem;
    Others1: TMenuItem;
    Orderby1: TMenuItem;
    Descending2: TMenuItem;
    Ascending2: TMenuItem;
    N2: TMenuItem;
    Others2: TMenuItem;
    Subtitles2: TMenuItem;
    Languages2: TMenuItem;
    Disks2: TMenuItem;
    Framerate2: TMenuItem;
    Resolution2: TMenuItem;
    AudioBitrate2: TMenuItem;
    AudioFormat2: TMenuItem;
    VideoBitrate2: TMenuItem;
    VideoFormat2: TMenuItem;
    Length2: TMenuItem;
    Borrower2: TMenuItem;
    MediaType2: TMenuItem;
    Media2: TMenuItem;
    DateAdded2: TMenuItem;
    MovieNumber2: TMenuItem;
    Country2: TMenuItem;
    Actors2: TMenuItem;
    Producer2: TMenuItem;
    Director2: TMenuItem;
    Decade2: TMenuItem;
    Year2: TMenuItem;
    Rating2: TMenuItem;
    TranslatedTitle2: TMenuItem;
    OriginalTitle2: TMenuItem;
    Category2: TMenuItem;
    PanelSearch: TPanel;
    BtnBack: TEncartaButton;
    EditSearch: TEdit;
    ComboSearch: TComboBox;
    BtnSearch: TImage;
    PanelClose: TPanel;
    BtnClose: TEncartaButton;
    PaintBox1: TPaintBox;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure NiceSideBar1Select(Sender: TObject; Index, SubIndex: Integer;
      Caption: string);
    procedure NiceGallery1Click(Sender: TObject; Index: Integer);
    procedure BtnOpenClick(Sender: TObject);
    procedure BtnEditClick(Sender: TObject);
    procedure BtnSettingsClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure BtnCategClick(Sender: TObject);
    procedure BtnPlayClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure BtnBackClick(Sender: TObject);
    procedure EditSearchKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure EditSearchClick(Sender: TObject);
    procedure BtnSearchClick(Sender: TObject);
    procedure BtnCloseClick(Sender: TObject);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure WebBrowser1CommandStateChange(Sender: TObject;
      Command: Integer; Enable: WordBool);
    procedure WebBrowser1BeforeNavigate2(Sender: TObject;
      const pDisp: IDispatch; var URL, Flags, TargetFrameName, PostData,
      Headers: OleVariant; var Cancel: WordBool);
    procedure Panel2Resize(Sender: TObject);
  private
    StarImages: array [0..10] of TMemoryStream;
    Blank: TMemoryStream;
    MovieHtml: TMemoryStream;
    Images: TStringList;
    CurrentItem: string;
    Container: TWBContainer;
    CatalogPath: string;
    GalleryPrepared: Boolean;
    OldTitleType: Integer;
    KBHook: HHook; {this intercepts keyboard input}
    ClearHistory: Boolean;
    NbFixedSortItems: Integer;
    NbFixedGroupItems: Integer;
    procedure ClearAllHistory;
    procedure PrepareGallery;
    procedure GroupMovies;
    procedure OpenItems(Movies: string; NoCache: Boolean = False);
    procedure LoadFile;
    procedure HandleNameSpace(Sender: TObject; Url: WideString; Stream: TMemoryStream;
      var Size: LongInt; var Found: Boolean);
    procedure LoadScheme(Section: string);
    function PrepareMovieImage(Prefix: string; Movie: TMovie; MaxWidth, MaxHeight: Integer): string;
    function LoadBitmap(Stream: TStream; Ext: string; MaxWidth, MaxHeight: Integer): TBitmap;
    function LoadThumbnail(Movie: TMovie): TBitmap;
    function LoadMovieBitmap(Movie: TMovie; MaxWidth, MaxHeight: Integer): TBitmap;
    function GetVideoPath(Movie: TMovie): string;
    procedure AlignButtons;

    procedure GroupToInt(var Value: Integer; GroupKind: TSortKind; GroupCFTag: string);
    procedure OrderToInt(var Value: Integer; SortKind: TSortKind; SortCFTag: string);
    procedure IntToGroup(Value: Integer; var GroupKind: TSortKind; var GroupCFTag: string);
    procedure IntToOrder(Value: Integer; var SortKind: TSortKind; var SortCFTag: string);
    procedure SetupGroupMenus;
    procedure FillGroupMenu;
    procedure FillSortMenu;
    procedure CheckAsc1;
    procedure CheckAsc2;
    procedure CheckGroup1;
    procedure CheckGroup2;
    procedure GroupClick1(Sender: TObject);
    procedure GroupClick2(Sender: TObject);
    procedure AscClick1(Sender: TObject);
    procedure AscClick2(Sender: TObject);
    procedure FillComboSearch;
    procedure Find(Term: string);
    procedure ClearImages;
    procedure LoadLanguage(Language: string);

   public
    List: TMovieList;
    TemplateHeader, TemplateFooter: string;
    ConfTemplate, ConfListTemplate, ConfUserVars: TStringList;
    ConfCatalogFile: string;
    ConfAutoOpen: Boolean;
    ConfTitleType: Integer;
    ConfColorScheme: string;
    ConfLanguage: string;
    ConfFullScreen: Boolean;
    ConfShowTopImages: Boolean;
    ConfAutoscroll: Boolean;
    ConfMaxImageWidth, ConfMaxImageHeight: Integer;
    ConfDateFormat: string;
    ConfExternalPlayer: string;
    ConfVideoPathField: string;
    ConfProtected: Boolean;
    ConfPassword: string;
    ConfShowOpenButton: Boolean;
    ConfShowEditButton: Boolean;
    ConfShowSettingsButton: Boolean;
    ConfShowPlayButton: Boolean;
    ConfShowSearch: Boolean;
    ConfMaximized: Boolean;
    ConfWindowLeft, ConfWindowTop: Integer;
    ConfWindowWidth, ConfWindowHeight: Integer;
    ConfMovieListWidth: Integer;
    ConfGroupKind: TSortKind;
    ConfGroupOrderDesc: Boolean;
    ConfGroupCFTag: string;
    ConfOrderKind: TSortKind;
    ConfOrderOrderDesc: Boolean;
    ConfOrderCFTag: string;
    function GetMovieTitle(Movie: TMovie; TitleType: TTitleType = ttNone): string;
    procedure LoadConfig;
    procedure SaveConfig;
    procedure ApplyConfig;
    function CheckPassword: Boolean;
    procedure UpdateLanguage;
  end;

var
  FormMain: TFormMain;
  function KeyboardHookProc(Code: Integer; WordParam: Word; LongParam: LongInt): LongInt; stdcall;

implementation

{$R *.dfm}
{$R Ratings.res}

uses
  JPEG, GifImage, UnitOpen, UnitSettings, UnitPassword, IniFiles, EncDec,
  UnitProgress, Clipbrd, ActiveX, TransUtils;

const
  DefaultTemplate =
    '<table width="100%" border="0" cellspacing="20">'#13#10 +
    '  <tr>'#13#10 +
    '    <td align="left" valign="top">$PICTURE</td>'#13#10 +
    '    <td>'#13#10 + 
    '      <p><b><font size="5">$ORIGINALTITLE</font></b>'#13#10 +
    '        ($YEAR)<br>'#13#10 +
    '        $RATINGIMG</p>'#13#10 +
    '      <p><b>Category<br></b>$CATEGORY</p>'#13#10 +
    '      <p><b>Length<br></b>$LENGTH Minutes</p>'#13#10 +
    '      <p><b>Directed by<br></b>$DIRECTOR</p>'#13#10 +
    '      <p><b>Actors<br></b>$ACTORS</p>'#13#10 +
    '      <p><b>Description<br></b>$DESCRIPTION</p>'#13#10 +
    '      <p><b>Comments<br></b>$COMMENTS</p>'#13#10 +
    '    </td>'#13#10 +
    '  </tr>'#13#10 +
    '</table>'#13#10 +
    '<p>&nbsp;</p>'#13#10;

  DefaultListTemplate =
    '<table width="100%" border="0" cellspacing="20">'#13#10 +
    '  <tr>'#13#10 +
    '    <td align="left" valign="top"><a href="$LINK">$SMALLPICTURE</a></td>'#13#10 +
    '    <td>'#13#10 + 
    '      <p>'#13#10 +
    '	<b><font size="4"><a href="$LINK">$TITLE</a></font></b> ($YEAR)'#13#10 +
    '	<br>'#13#10 +
    '	$RATINGIMG'#13#10 +
    '      </p>'#13#10 +
    '      <p>'#13#10 +
    '        <b>Category: </b>$CATEGORY<br>'#13#10 +
    '        <b>Director: </b>$DIRECTOR<br>'#13#10 +
    '        <b>Actors: </b>$ACTORS<br>'#13#10 +
    '      </p>'#13#10 +
    '    </td>'#13#10 +
    '  </tr>'#13#10 +
    '</table>'#13#10 +
    '<hr noshade>'#13#10;

function KeyboardHookProc(Code: Integer; WordParam: Word; LongParam: LongInt) : LongInt;
 begin
  if (WordParam = 80) and (not FormMain.EditSearch.Focused) then  // 80 = 'p'
    FormMain.BtnPlayClick(nil);

  Result:=0;
  {To prevent Windows from passing the keystrokes
  to the target window, the Result value must
  be a nonzero value.}
 end;

procedure TFormMain.FormCreate(Sender: TObject);

  procedure LoadResource(Stream: TMemoryStream; ResName: string);
  var
    Res: TResourceStream;
  begin
    Res := TResourceStream.Create(HInstance, ResName, RT_RCDATA);
    try
      Stream.CopyFrom(Res, 0);
    finally
      Res.Free;
    end;
  end;

var
  x: Integer;
  Mem: TMemoryStream;

begin
  ClearHistory := False;
  NbFixedGroupItems := Others1.Count;
  NbFixedSortItems := Others2.Count;

  KBHook:=SetWindowsHookEx(WH_KEYBOARD,
            {callback >} @KeyboardHookProc,
                           HInstance,
                           GetCurrentThreadId()) ;

  ConfGroupKind := skNone;
  ConfGroupOrderDesc := False;
  ConfGroupCFTag := '';
  ConfOrderKind := DefaultSortKind;
  ConfOrderOrderDesc := False;
  ConfOrderCFTag := '';
  SetupGroupMenus;

  Container := TWBContainer.Create(WebBrowser1);
  Container.Show3DBorder := False;

  List := TMovieList.Create;

  Label1.Left := Panel1.Width - Label1.Width - 20;
  Label1.Anchors := [akTop, akRight];
  Label2.Left := Panel1.Width - Label2.Width - 20;
  Label2.Anchors := [akTop, akRight];

  for x := 0 to 10 do
  begin
    Mem := TMemoryStream.Create;
    LoadResource(Mem, 'stars' + Format('%.2d', [x]));
    StarImages[x] := Mem;
  end;
  Blank := TMemoryStream.Create;
  LoadResource(Blank, 'blank');

  Images := TStringList.Create;
  //Images.Sorted := True;
  //Images.Duplicates := dupAccept;
  MovieHtml := TMemoryStream.Create;
  RegisterNameSpace(HandleNameSpace);

  ConfTemplate := TStringList.Create;
  ConfListTemplate := TStringList.Create;
  ConfUserVars := TStringList.Create;
  LoadConfig;

  Left := ConfWindowLeft;
  Top := ConfWindowTop;
  Width := ConfWindowWidth;
  Height := ConfWindowHeight;
  Panel2.Width := ConfMovieListWidth;
  
  if ConfFullScreen
    then BorderStyle := bsNone;
  PanelClose.Visible := ConfFullScreen; 
  if ConfFullScreen or ConfMaximized
    then WindowState := wsMaximized;

  LoadScheme(ConfColorScheme);
  LoadLanguage(ConfLanguage);

  AlignButtons;

  // Default page
  OpenItems('DefaultPage');

  if ConfAutoOpen
    then LoadFile;

end;

procedure TFormMain.FormDestroy(Sender: TObject);
var
  x: Integer;
  Mem: TMemoryStream;
begin
  if (not ConfFullScreen) then
  begin
    ConfMaximized := WindowState = wsMaximized;
    if not ConfMaximized then
    begin
      ConfWindowLeft := Left;
      ConfWindowTop := Top;
      ConfWindowWidth := Width;
      ConfWindowHeight := Height;
    end;
  end;
  ConfMovieListWidth := Panel2.Width;
  SaveConfig;
  UnregisterNameSpace;
  MovieHtml.Free;
  for x := 0 to 10 do
  begin
    Mem := StarImages[x];
    Mem.Free;
  end;
  Blank.Free;
  List.Free;
  Container.Free;
  ConfUserVars.Free;
  ConfTemplate.Free;
  ConfListTemplate.Free;
  ClearImages;
  Images.Free;
  {unhook the keyboard interception}
  UnHookWindowsHookEx(KBHook) ;
end;

procedure TFormMain.LoadFile;
begin
  NiceGallery1.Active := False;
  NiceGallery1.Items.Clear;
  NiceSideBar1.Items.Clear;
  List.CustomFieldsProperties.Clear;
  List.Clear;
  ClearImages;

  if not FileExists(ConfCatalogFile) then
  begin
    Label4.Caption := _('No Movie');
    Label5.Caption := _('Open a catalog file first ...');
    Exit;
  end;

  List.LoadFromFile(ConfCatalogFile);
  CatalogPath := ExtractFilePath(ConfCatalogFile);

  if (List.MovieProperties.strName <> '')
    then Label1.Caption := List.MovieProperties.strName + '''s'
    else Label1.Caption := _('My Private');

  case List.Count of
    0: Label4.Caption := _('No Movie');
    1: Label4.Caption := _('1 Movie');
  else
    Label4.Caption := IntToStr(List.Count) + ' ' + _('Movies');
  end;
  Label5.Caption := ConfCatalogFile;

  FillGroupMenu;
  FillSortMenu;

  CheckGroup1;
  CheckGroup2;
  
  FillComboSearch;

  GroupMovies;

  GalleryPrepared := False;
  if ConfShowTopImages
    then PrepareGallery;
  NiceGallery1.Visible := ConfShowTopImages;  
  NiceGallery1.Active := ConfShowTopImages and ConfAutoscroll;

  if (NiceSideBar1.Items.Count > 0) then
  begin
    if (NiceSideBar1.Items[0].Items.Count > 0) then
    begin
      NiceSideBar1.ItemIndex := 0;
      NiceSideBar1.SubItemIndex := 0;
      NiceSideBar1Select(nil, 0, 0, '');
    end;
  end;

end;

function TFormMain.LoadThumbnail(Movie: TMovie): TBitmap;
const
  MaxThumbWidth = 100;
  MaxThumbHeight = 100;
var
  Ext: string;
  Stream: TStream;
begin
  Result := nil;
  Ext := ExtractFileExt(Movie.strPicture);
  if Assigned(Movie.Picture) then
  begin
    Stream := Movie.Picture;
    Result := LoadBitmap(Stream, Ext, MaxThumbWidth, MaxThumbHeight);
  end else
  if (ExtractFileDrive(Movie.strPicture) <> '') then
  begin
    Stream := TFileStream.Create(Movie.strPicture, fmOpenRead);
    try
      Result := LoadBitmap(Stream, Ext, MaxThumbWidth, MaxThumbHeight);
    finally
      Stream.Free;
    end;
  end else
  if (Movie.strPicture <> '') then
  begin
    Stream := TFileStream.Create(CatalogPath + Movie.strPicture, fmOpenRead);
    try
      Result := LoadBitmap(Stream, Ext, MaxThumbWidth, MaxThumbHeight);
    finally
      Stream.Free;
    end;
  end;
end;

function TFormMain.LoadBitmap(Stream: TStream; Ext: string; MaxWidth, MaxHeight: Integer): TBitmap;
const
  Scales: array [0..3] of TJPEGScale = (jsEighth, jsQuarter, jsHalf, jsFullSize);
var
  Bmp, Thumb: TBitmap;
  Jpg: TJPEGImage;
  Gif: TGifImage;
  Png: TPngObject;
  x1, y1: Single;
  w, h: Integer;
  i: Integer;
begin
  Result := nil;
  Stream.Seek(0, soFromBeginning);
  Bmp := TBitmap.Create;
  try
    if SameText(Ext, '.jpg') then
    begin
      Jpg := TJpegImage.Create;
      try
        for i := 0 to 3 do
        begin
          Jpg.Scale := Scales[i];
          Jpg.LoadFromStream(Stream);
          if (Jpg.Width >= MaxWidth) or (Jpg.Height >= MaxHeight)
            then Break;
          Stream.Seek(0, soFromBeginning);
        end;
        Bmp.Assign(Jpg);
      finally
        Jpg.Free;
      end;  
    end else
    if SameText(Ext, '.gif') then
    begin
      Gif := TGifImage.Create;
      try
        Gif.LoadFromStream(Stream);
        Bmp.Assign(Gif);
      finally
        Gif.Free;
      end;  
    end else
    if SameText(Ext, '.png') then
    begin
      Png := TPngObject.Create;
      try
        Png.LoadFromStream(Stream);
        Bmp.Assign(Png);
      finally
        Png.Free;
      end;  
    end else
    if SameText(Ext, '.bmp') then
    begin
      Bmp.LoadFromStream(Stream);
    end else
    begin
      Bmp.Free;
      Exit;
    end;
  except
    Bmp.Free;
    Exit;
  end;
  Thumb := TBitmap.Create;
  if (Bmp.Width > MaxWidth) or (Bmp.Height > MaxHeight) then
  begin
    x1 := MaxWidth / Bmp.Width;
    y1 := MaxHeight / Bmp.Height;
    if (x1 > y1) then
    begin
      h := MaxHeight;
      w := Round(Bmp.Width * y1);
    end else
    begin
      w := MaxWidth;
      h := Round(Bmp.Height*x1);
    end;
    Thumb.Width := w;
    Thumb.Height := h;
    Thumb.Canvas.StretchDraw(Rect(0, 0, w, h), Bmp);
  end else
    Thumb.Assign(Bmp);
  Bmp.Free; //MV - Fix a "BIG" memory leak
  Result := Thumb;
end;

function TFormMain.GetMovieTitle(Movie: TMovie; TitleType: TTitleType): string;
begin
  Result := _('< no title >');
  if TitleType = ttNone then
    TitleType := ConfTitleType;
  if TitleType = ttNone then
    TitleType := ttOriginal;
  if (TitleType = ttOriginal) then
  begin
    if (Movie.strOriginalTitle <> '') then
      Result := Movie.strOriginalTitle
    else if (Movie.strTranslatedTitle <> '') then
      Result := Movie.strTranslatedTitle;
  end else if (TitleType = ttTranslated) then
  begin
    if (Movie.strTranslatedTitle <> '') then
      Result := Movie.strTranslatedTitle
    else if (Movie.strOriginalTitle <> '') then
      Result := Movie.strOriginalTitle;
  end else
  begin
    if (Movie.strMedia <> '') then
      Result := Movie.strMedia
    else if (Movie.strOriginalTitle <> '') then
      Result := Movie.strOriginalTitle
    else if (Movie.strTranslatedTitle <> '') then
      Result := Movie.strTranslatedTitle;
  end;
end;

procedure TFormMain.PrepareGallery;
var
  Item: TGalleryItem;
  Movie: TMovie;
  x: Integer;
  Bmp: TBitmap;
begin
  FormProgress.Show;
  NiceGallery1.Items.BeginUpdate;
  try
    FormProgress.ProgressBar1.Position := 0;
    FormProgress.ProgressBar1.Max := List.Count;
    for x := 0 to List.Count-1 do
    begin
      Movie := TMovie(List[x]);
      try
        Bmp := LoadThumbnail(Movie);
      except
        Bmp := nil;
      end;
      if Assigned(Bmp) then
      begin
        Item := NiceGallery1.Items.Add;
        Item.Hint := GetMovieTitle(Movie);
        Item.Tag := Integer(Movie);
        Item.Bitmap.Assign(Bmp);
        Bmp.Free;
      end;
      FormProgress.ProgressBar1.StepIt;
      FormProgress.Update;
      Application.ProcessMessages;
    end;
  finally
    NiceGallery1.Items.EndUpdate;
    FormProgress.Hide;
  end;
  Application.BringToFront;
  GalleryPrepared := True;
end;

function TFormMain.LoadMovieBitmap(Movie: TMovie; MaxWidth, MaxHeight: Integer): TBitmap;
var
  Ext: string;
  Stream: TStream;
begin
  Result := nil;
  Ext := ExtractFileExt(Movie.strPicture);
  if Assigned(Movie.Picture) then
  begin
    Stream := Movie.Picture;
    Result := LoadBitmap(Stream, Ext, MaxWidth, MaxHeight);
  end else
  if (ExtractFileDrive(Movie.strPicture) <> '') then
  begin
    Stream := TFileStream.Create(Movie.strPicture, fmOpenRead);
    try
      Result := LoadBitmap(Stream, Ext, MaxWidth, MaxHeight);
    finally
      Stream.Free;
    end;
  end else
  if (Movie.strPicture <> '') then
  begin
    Stream := TFileStream.Create(CatalogPath + Movie.strPicture, fmOpenRead);
    try
      Result := LoadBitmap(Stream, Ext, MaxWidth, MaxHeight);
    finally
      Stream.Free;
    end;
  end;
end;

procedure TFormMain.ClearImages;
var
  x: Integer;
  m: TMemoryStream;
begin
  for x := 0 to Images.Count-1 do
  begin
    m := TMemoryStream(Images.Objects[x]);
    m.Free;
  end;
  Images.Clear;
end;

function TFormMain.PrepareMovieImage(Prefix: string; Movie: TMovie; MaxWidth, MaxHeight: Integer): string;
var
  Str: string;
  Jpg: TJpegImage;
  Bmp: TBitmap;
  Stream: TMemoryStream;
  FileName: string;
begin
  Stream := TMemoryStream.Create;
  FileName := 'image-' + Prefix + '-' + IntToStr(Integer(Movie));
  if (MaxWidth > 0) and (MaxHeight > 0) then
  begin
    try
      Bmp := LoadMovieBitmap(Movie, MaxWidth, MaxHeight);
    except
      Bmp := nil;
    end;
    if Assigned(Bmp) then
    begin
      Jpg := TJPEGImage.Create;
      try
        Jpg.Assign(Bmp);
        Jpg.SaveToStream(Stream);
        Result := FileName + '.jpg';
      finally
        Jpg.Free;
      end;
      Bmp.Free;
    end else
    begin
      Stream.CopyFrom(Blank, 0);
      Result := FileName + '.gif';
    end;
  end else
  if Assigned(Movie.Picture) then
  begin
    Movie.Picture.SaveToStream(Stream);
    Result := FileName + Movie.strPicture;
  end else
  if (Movie.strPicture <> '') then
  begin
    if (ExtractFileDrive(Movie.strPicture) <> '')
      then Str := Movie.strPicture
      else Str := CatalogPath + Movie.strPicture;
    try
      Stream.LoadFromFile(Str);
      Result := FileName + ExtractFileExt(Str);
    except
      Stream.CopyFrom(Blank, 0);
      Result := FileName + '.gif';
    end;
  end else
  begin
    Stream.CopyFrom(Blank, 0);
    Result := FileName + '.gif';
  end;
  Images.AddObject(Result, Stream);
end;

procedure TFormMain.GroupMovies;
var
  Title: string;
  GroupKind, OrderKind: TSortKind;
  GroupCFTag, OrderCFTag: string;
  i: Integer;
begin
  GroupToInt(i, ConfGroupKind, ConfGroupCFTag);
  IntToGroup(i, GroupKind, GroupCFTag);
  OrderToInt(i, ConfOrderKind, ConfOrderCFTag);
  IntToOrder(i, OrderKind, OrderCFTag);
  if List <> nil then
    GroupAndSortMovies(List, NiceSideBar1, ConfTitleType,
      GroupKind, ConfGroupOrderDesc, List.CustomFieldsProperties.GetField(GroupCFTag),
      OrderKind, ConfOrderOrderDesc, List.CustomFieldsProperties.GetField(OrderCFTag), Title)
  else
    GroupAndSortMovies(List, NiceSideBar1, ConfTitleType,
      GroupKind, ConfGroupOrderDesc, nil,
      OrderKind, ConfOrderOrderDesc, nil, Title);
  LabelGroup.Caption := Title;
end;

procedure TFormMain.NiceSideBar1Select(Sender: TObject; Index,
  SubIndex: Integer; Caption: String);
var
  Movies: string;
begin
  if (Index = -1) or (SubIndex = -1)
    then Exit;
  if (NiceSideBar1.Tag = 1)
    then Movies := TStringList(NiceSideBar1.Items[Index].Tag).Strings[SubIndex]
    else Movies := IntToStr(Integer(NiceSideBar1.Items[Index].Items.Objects[SubIndex]));
  CurrentItem := Movies;
  OpenItems(Movies, Sender = nil);
end;

procedure TFormMain.NiceGallery1Click(Sender: TObject; Index: Integer);
var
  Movies: string;
begin
  if (Index = -1)
    then Exit;
  NiceSideBar1.ItemIndex := -1;
  NiceSideBar1.SubItemIndex := -1;
  Movies := IntToStr(NiceGallery1.Items[Index].Tag);
  CurrentItem := Movies;
  OpenItems(Movies);
end;

procedure TFormMain.OpenItems(Movies: string; NoCache: Boolean);
var
  Flags: OLEVariant;
begin
  try
    Flags := 0;
    if NoCache then
    begin
      ClearAllHistory;
      Flags := 2; // Replace current page by this one
    end;
    WebBrowser1.Navigate('movie://localhost/movie[' + Movies + '].html', Flags);
  except
  end;
end;

procedure TFormMain.HandleNameSpace(Sender: TObject; Url: WideString;
  Stream: TMemoryStream; var Size: Integer; var Found: Boolean);
  
  function IntToStrOrBlank(Num: Integer): string;
  begin
    if (Num <> -1)
      then Result := IntToStr(Num)
      else Result := '';
  end;

  function NewLineToBR(Str: string): string;
  begin
    Result := StringReplace(Str, #13#10, '<br>', [rfReplaceAll]);
    Result := StringReplace(Result, #13, '<br>', [rfReplaceAll]);
    Result := StringReplace(Result, #10, '<br>', [rfReplaceAll]);
  end;

const
  YesNo: array [Boolean] of string = ('No', 'Yes');
var
  TempStr, Subs: string;
  Str, FileName, FileExt: string;
  Temp: TMemoryStream;
  Movie: TMovie;
  t: TStringList;
  i, j:integer;
  IsMovie: Boolean;
  Img: TMemoryStream;
  StrName, StrValue: string;
  n: Integer;
  Tag, Value: string;

begin
  if ClearHistory then
  begin
    TempStr := '<br>';
    Stream.Write(TempStr[1], Length(TempStr));
    Size := Stream.Size;
    Found := True;
    Exit;
  end;

  Str := StringReplace(Url, 'movie://localhost/', '', [rfReplaceAll]);
  FileName := ExtractFileName(Str);
  FileExt := ExtractFileExt(Str);
  if (Copy(FileName, 1, 5) = 'movie') then
  begin
    //Alex

    t := TStringList.Create;

    //expected format of the url: 'movie[a,b,..,n].html', display all movies a,..,n
    ExtractStrings(['[',']',','], [' '], PChar(FileName), t);

    Stream.Clear;

    TempStr := TemplateHeader;

    for i := 1 to t.Count - 2 do // Memo1[0] is the part of te filename before the '[', so we skip it, same for the extention
    begin

      IsMovie := False;
      Movie := nil;

      if t[i] = 'DefaultPage' then //default page
        Subs := '<br>'
      else if t[i] = '...' then //too many results
        Subs := '<br><br><center><big><b>...</b></big><br><br></center>'
      else if (t[i] <> '') and (t[i][1] = '"') then //movie not found during search
        Subs := '<br><br><br><br><br><br><center>' +
          StringReplace(_('Sorry, your search for <b>$SEARCH</b> did not return any results.'), '$SEARCH', t[i], [rfReplaceAll]) +
          '<br>' + _('Please try again or select a movie from the menu.') + '</center>'
      else
        try
          Movie := TMovie(StrToInt(t[i]));
          IsMovie := True;
        except
          Subs := 'Unexpected Error... ' + t[i];
        end;

      if IsMovie then
      begin

 {
        //Alex:imdb hack, should be config able
        Temp2 := Movie.strActors;
        Temp2 := StringReplace(Temp2, '<a href="/character/', '<a href="http://imdb.com/character/', [rfReplaceAll]);
 }
        //use detailed template for single movie and short one for list:
        if (t.Count = 3)
          then begin
               Subs := ConfTemplate.Text;
               CurrentItem := IntToStr(Integer(Movie));      // Modif L LEFFET
               end
          else Subs := ConfListTemplate.Text;

        Subs := StringReplace(Subs, '$LINK', 'movie://localhost/movie[' + IntToStr(Integer(Movie)) + '].html', [rfReplaceAll]);
        Subs := StringReplace(Subs, '$NUMBER', IntToStrOrBlank(Movie.iNumber), [rfReplaceAll]);
        Subs := StringReplace(Subs, '$MEDIATYPE', Movie.strMediaType, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$MEDIA', Movie.strMedia, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$SOURCE', Movie.strSource, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$DATE', FormatDateTime(ConfDateFormat, Movie.iDate), [rfReplaceAll]);
        Subs := StringReplace(Subs, '$BORROWER', Movie.strBorrower, [rfReplaceAll]);

        Subs := StringReplace(Subs, '$TITLE', GetMovieTitle(Movie), [rfReplaceAll]);
        Subs := StringReplace(Subs, '$ORIGINALTITLE', Movie.strOriginalTitle, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$TRANSLATEDTITLE', Movie.strTranslatedTitle, [rfReplaceAll]);

        Subs := StringReplace(Subs, '$DIRECTOR', Movie.strDirector, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$PRODUCER', Movie.strProducer, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$COUNTRY', Movie.strCountry, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$CATEGORY', Movie.strCategory, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$YEAR', IntToStrOrBlank(Movie.iYear), [rfReplaceAll]);
        Subs := StringReplace(Subs, '$LENGTH', IntToStrOrBlank(Movie.iLength), [rfReplaceAll]);

        Subs := StringReplace(Subs, '$ACTORS', NewLineToBR(Movie.strActors), [rfReplaceAll]);

        Subs := StringReplace(Subs, '$URL', Movie.strURL, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$DESCRIPTION', NewLineToBR(Movie.strDescription), [rfReplaceAll]);
        Subs := StringReplace(Subs, '$COMMENTS', NewLineToBR(Movie.strComments), [rfReplaceAll]);
        Subs := StringReplace(Subs, '$VIDEOFORMAT', Movie.strVideoFormat, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$BITRATE', IntToStrOrBlank(Movie.iVideoBitrate), [rfReplaceAll]);
        Subs := StringReplace(Subs, '$AUDIOFORMAT', Movie.strAudioFormat, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$AUDIOBITRATE', IntToStrOrBlank(Movie.iAudioBitrate), [rfReplaceAll]);
        Subs := StringReplace(Subs, '$RESOLUTION', Movie.strResolution, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$FRAMERATE', Movie.strFramerate, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$LANGUAGES', Movie.strLanguages, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$SUBTITLES', Movie.strSubtitles, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$SIZE', Movie.strSize, [rfReplaceAll]);
        Subs := StringReplace(Subs, '$DISKS', IntToStrOrBlank(Movie.iDisks), [rfReplaceAll]);
        Subs := StringReplace(Subs, '$CHECKED', _(YesNo[Movie.bChecked]), [rfReplaceAll]);

        Subs := StringReplace(Subs, '$PICTURE', '<img src="' + PrepareMovieImage('large', Movie, ConfMaxImageWidth, ConfMaxImageHeight) + '">', [rfReplaceAll]);
        Subs := StringReplace(Subs, '$SMALLPICTURE', '<img src="' + PrepareMovieImage('small', Movie, 200, 200) + '">', [rfReplaceAll]);

        Subs := StringReplace(Subs, '$VIDEOPATH', GetVideoPath(Movie), [rfReplaceAll]);

        n := Round(Max(0, Min(100, Movie.iRating)) / 10);
        Subs := StringReplace(Subs, '$RATINGIMG', '<img src="rating' + Format('%.2d', [n]) + '.gif">', [rfReplaceAll]);
        Subs := StringReplace(Subs, '$RATING', FormatFloat('0.#', Movie.iRating / 10), [rfReplaceAll]);

        if (List <> nil) and (List.CustomFieldsProperties <> nil) then
          with List.CustomFieldsProperties do
            for n := Count-1 downto 0  do
            begin
              Tag := Objects[n].FieldTag;
              Value := Movie.CustomFields.GetFieldValue(Tag, False, True);
              try
                if (Value <> '') then
                begin
                  if (Objects[n].FieldType = ftDate) then
                    Value := FormatDateTime(ConfDateFormat, StrToDate(Value));
                end;
              except
              end;
              Subs := StringReplace(Subs, '$CF_'+ UpperCase(Tag), Value, [rfReplaceAll]);
            end;
      end;

      if (ConfUserVars.Count > 1) then
      begin
        for j := 0 to ConfUserVars.Count-1 do
        begin
          StrName := ConfUserVars.Names[j];
          StrValue := ConfUserVars.Values[StrName];
          StrName := StringReplace(StrName, '&equal;', '=', [rfReplaceAll]);
          StrValue := StringReplace(StrValue, '&equal;', '=', [rfReplaceAll]);
          if (StrName <> '')
            then Subs := StringReplace(Subs, StrName, StrValue, [rfReplaceAll]);
        end;
      end;

      TempStr := TempStr+ Subs;

    end;

    TempStr := TempStr + TemplateFooter;

    Stream.Write(TempStr[1], Length(TempStr));
    Size := Stream.Size;

    t.Free;

    Found := True;

  end else
  if (Copy(FileName, 1, 5) = 'image') then
  begin
    for i := 0 to Images.Count-1 do
    begin
      if SameText(FileName, Images[i]) then
      begin
        Img := TMemoryStream(Images.Objects[i]);
        Stream.CopyFrom(Img, 0);
        Size := Img.Size;
        Found := True;
        Break;
      end;
    end;
  end else
  if (Copy(FileName, 1, 6) = 'rating') and (Copy(FileName, 9, 4) = '.gif') then
  begin
    n := StrToIntDef(Copy(FileName, 7, 2), 0);
    n := Max(0, Min(10, n));
    Temp := StarImages[n];
    Stream.CopyFrom(Temp, 0);
    Size := Temp.Size;
    Found := True;
  end;
  if not Found then
  begin
    if FileExists(ExtractFilePath(Application.ExeName) + FileName) then
    begin
      Temp := TMemoryStream.Create;
      try
        try
          Temp.LoadFromFile(ExtractFilePath(Application.ExeName) + FileName);
          Stream.CopyFrom(Temp, 0);
          Size := Temp.Size;
          Found := True;
        except
          //
        end;
      finally
        Temp.Free;
      end;
    end;
  end;
end;

procedure TFormMain.BtnOpenClick(Sender: TObject);
begin
  if not CheckPassword
    then Exit;
  FormOpen.EditFileName.Text := ConfCatalogFile;
  FormOpen.CheckAutoOpen.Checked := ConfAutoOpen;
  FormOpen.ShowModal;
  if FormOpen.Done then
  begin
    ConfCatalogFile := FormOpen.EditFileName.Text;
    ConfAutoOpen := FormOpen.CheckAutoOpen.Checked;
    LoadFile;
    SaveConfig;
  end;
end;

procedure TFormMain.BtnEditClick(Sender: TObject);
begin
  if not CheckPassword
    then Exit;
  if FileExists(ConfCatalogFile)
    then ShellExecute(0, 'open', PChar(ConfCatalogFile), nil, nil, SW_SHOW);
end;

procedure TFormMain.BtnPlayClick(Sender: TObject);
var
  n: Integer;
  Path: string;
begin
  n := StrToIntDef(CurrentItem, -1);
  if (n = -1)
    then Exit;
  Path := GetVideoPath(TMovie(n));
  if (Path = '')
    then Exit;
  if (ConfExternalPlayer = '')
    then ShellExecute(0, 'open', PChar('"' + Path + '"'), nil, nil, SW_SHOW)
    else ShellExecute(0, 'open', PChar('"' + ConfExternalPlayer + '"'),
           PChar('"' + Path + '"'), nil, SW_SHOW);
end;

//Alex
procedure TFormMain.BtnBackClick(Sender: TObject);
begin
  try
    WebBrowser1.GoBack;
  except
    //
  end;
end;

procedure TFormMain.BtnSettingsClick(Sender: TObject);
begin
  if not CheckPassword
    then Exit;
  OldTitleType := ConfTitleType;
  FormSettings.ShowModal;
end;

function TFormMain.GetVideoPath(Movie: TMovie): string;
begin
  Result := '';
  if not Assigned(Movie) or not Assigned(List)
    then Exit;
  if (ConfVideoPathField = '')
    then Exit;
  if SameText(ConfVideoPathField, 'Media')
    then Result := Movie.strMedia else
  if SameText(ConfVideoPathField, 'MediaType')
    then Result := Movie.strMediaType else
  if SameText(ConfVideoPathField, 'Source')
    then Result := Movie.strSource else
  if SameText(ConfVideoPathField, 'Borrower')
    then Result := Movie.strBorrower else
  if SameText(ConfVideoPathField, 'OriginalTitle')
    then Result := Movie.strOriginalTitle else
  if SameText(ConfVideoPathField, 'TranslatedTitle')
    then Result := Movie.strTranslatedTitle else
  if SameText(ConfVideoPathField, 'Director')
    then Result := Movie.strDirector else
  if SameText(ConfVideoPathField, 'Producer')
    then Result := Movie.strProducer else
  if SameText(ConfVideoPathField, 'Country')
    then Result := Movie.strCountry else
  if SameText(ConfVideoPathField, 'Category')
    then Result := Movie.strCategory else
  if SameText(ConfVideoPathField, 'Actors')
    then Result := Movie.strActors else
  if SameText(ConfVideoPathField, 'URL')
    then Result := Movie.strURL else
  if SameText(ConfVideoPathField, 'Description')
    then Result := Movie.strDescription else
  if SameText(ConfVideoPathField, 'Comments')
    then Result := Movie.strComments else
  if SameText(ConfVideoPathField, 'VideoFormat')
    then Result := Movie.strVideoFormat else
  if SameText(ConfVideoPathField, 'AudioFormat')
    then Result := Movie.strAudioFormat else
  if SameText(ConfVideoPathField, 'Resolution')
    then Result := Movie.strResolution else
  if SameText(ConfVideoPathField, 'Framerate')
    then Result := Movie.strFramerate else
  if SameText(ConfVideoPathField, 'Languages')
    then Result := Movie.strLanguages else
  if SameText(ConfVideoPathField, 'Subtitles')
    then Result := Movie.strSubtitles else
  if SameText(ConfVideoPathField, 'Size')
    then Result := Movie.strSize else
  begin
    Result := Movie.CustomFields.GetFieldValue(ConfVideoPathField, False);
  end;
end;

procedure TFormMain.LoadConfig;
const
  Section1 = 'Config';
  Section2 = 'Template';
  Section3 = 'ListTemplate';
  Section4 = 'UserVars';
var
  Ini: TIniFile;
  x, n: Integer;
  StrName, StrValue: string;
begin
  Ini := TIniFile.Create(ExtractFilePath(Application.ExeName) + 'Config.ini');
  try

    // load general config
    ConfCatalogFile := Ini.ReadString(Section1, 'CatalogFile', '');
    ConfAutoOpen := Ini.ReadBool(Section1, 'AutoOpen', False);
    ConfTitleType := Ini.ReadInteger(Section1, 'TitleType', 0);
    ConfColorScheme := Ini.ReadString(Section1, 'ColorScheme', 'Default');
    ConfLanguage := Ini.ReadString(Section1, 'Language', 'English');
    ConfFullScreen := Ini.ReadBool(Section1, 'FullScreen', False);
    ConfShowTopImages := Ini.ReadBool(Section1, 'ShowTopImages', True);
    ConfAutoscroll := Ini.ReadBool(Section1, 'AutoScroll', True);
    ConfMaxImageWidth := Ini.ReadInteger(Section1, 'MaxImageWidth', 400);
    ConfMaxImageHeight := Ini.ReadInteger(Section1, 'MaxImageHeight', 400);
    ConfDateFormat := Ini.ReadString(Section1, 'DateFormat', 'mm/dd/yyyy');
    ConfExternalPlayer := Ini.ReadString(Section1, 'ExternalPlayer', '');
    ConfVideoPathField := Ini.ReadString(Section1, 'VideoPathField', '');
    ConfProtected := Ini.ReadBool(Section1, 'Protected', False);
    ConfPassword := DecodeString(Ini.ReadString(Section1, 'Path', ''));
    ConfShowOpenButton := Ini.ReadBool(Section1, 'ShowOpenButton', True);
    ConfShowEditButton := Ini.ReadBool(Section1, 'ShowEditButton', True);
    ConfShowSettingsButton := Ini.ReadBool(Section1, 'ShowSettingsButton', True);
    ConfShowPlayButton := Ini.ReadBool(Section1, 'ShowPlayButton', True);
    ConfShowSearch := Ini.ReadBool(Section1, 'ShowSearch', True);
    ConfMaximized := Ini.ReadBool(Section1, 'Maximized', True);
    ConfWindowLeft := Ini.ReadInteger(Section1, 'Left', (Screen.Width - 800) div 2);
    ConfWindowTop := Ini.ReadInteger(Section1, 'Top', (Screen.Height - 600) div 2);
    ConfWindowWidth := Ini.ReadInteger(Section1, 'Width', 800);
    ConfWindowHeight := Ini.ReadInteger(Section1, 'Height', 600);
    ConfMovieListWidth := Ini.ReadInteger(Section1, 'MovieListWidth', 249);

    ConfGroupKind := TSortKind(Ini.ReadInteger(Section1, 'LastGroup', Ord(skNone)));
    ConfGroupOrderDesc := Ini.ReadBool(Section1, 'LastGroupDesc', False);
    ConfGroupCFTag := Ini.ReadString(Section1, 'LastGroupCFTag', '');
    ConfOrderKind := TSortKind(Ini.ReadInteger(Section1, 'LastOrder', Ord(DefaultSortKind)));
    ConfOrderOrderDesc := Ini.ReadBool(Section1, 'LastOrderDesc', False);
    ConfOrderCFTag := Ini.ReadString(Section1, 'LastOrderCFTag', '');

    CheckGroup1;
    CheckGroup2;
    CheckAsc1;
    CheckAsc2;

    // load template
    ConfTemplate.Clear;
    n := Ini.ReadInteger(Section2, 'Count', 0);
    for x := 1 to n
      do ConfTemplate.Add(Ini.ReadString(Section2, 'Line' + IntToStr(x), ''));
    ConfTemplate.Text := Trim(ConfTemplate.Text);
    if (ConfTemplate.Count = 0)
      then ConfTemplate.Text := DefaultTemplate;

    // load list template
    ConfListTemplate.Clear;
    n := Ini.ReadInteger(Section3, 'Count', 0);
    for x := 1 to n
      do ConfListTemplate.Add(Ini.ReadString(Section3, 'Line' + IntToStr(x), ''));
    ConfListTemplate.Text := Trim(ConfListTemplate.Text);
    if (ConfListTemplate.Count = 0)
      then ConfListTemplate.Text := DefaultListTemplate;

    // load user vars  
    ConfUserVars.Clear;
    n := Ini.ReadInteger(Section4, 'Count', 0);
    for x := 1 to n do
    begin
      StrName := Ini.ReadString(Section4, 'Name' + IntToStr(x), '');
      StrValue := Ini.ReadString(Section4, 'Value' + IntToStr(x), '');
      if (StrName <> '')
        then ConfUserVars.Values[StrName] := StrValue;
    end;

  finally
    Ini.Free;
  end;
end;

procedure TFormMain.SaveConfig;
const
  Section1 = 'Config';
  Section2 = 'Template';
  Section3 = 'ListTemplate';
  Section4 = 'UserVars';
var
  Ini: TIniFile;
  x: Integer;
  StrName, StrValue: string;
begin
  // if we load from CD/DVD Drive, just skip writing config
  if (GetDriveType(PChar(ExtractFileDrive(Application.ExeName))) = DRIVE_CDROM)
    then Exit;

  Ini := TIniFile.Create(ExtractFilePath(Application.ExeName) + 'Config.ini');
  try

    // save general config
    Ini.WriteString(Section1, 'CatalogFile', ConfCatalogFile);
    Ini.WriteBool(Section1, 'AutoOpen', ConfAutoOpen);
    Ini.WriteInteger(Section1, 'TitleType', ConfTitleType);
    Ini.WriteString(Section1, 'ColorScheme', ConfColorScheme);
    Ini.WriteString(Section1, 'Language', ConfLanguage);
    Ini.WriteBool(Section1, 'FullScreen', ConfFullScreen);
    Ini.WriteBool(Section1, 'ShowTopImages', ConfShowTopImages);
    Ini.WriteBool(Section1, 'AutoScroll', ConfAutoscroll);
    Ini.WriteInteger(Section1, 'MaxImageWidth', ConfMaxImageWidth);
    Ini.WriteInteger(Section1, 'MaxImageHeight', ConfMaxImageHeight);
    Ini.WriteString(Section1, 'DateFormat', ConfDateFormat);
    Ini.WriteString(Section1, 'ExternalPlayer', ConfExternalPlayer);
    Ini.WriteString(Section1, 'VideoPathField', ConfVideoPathField);
    Ini.WriteBool(Section1, 'Protected', ConfProtected);
    Ini.WriteString(Section1, 'Path', EncodeString(ConfPassword));
    Ini.WriteBool(Section1, 'ShowOpenButton', ConfShowOpenButton);
    Ini.WriteBool(Section1, 'ShowEditButton', ConfShowEditButton);
    Ini.WriteBool(Section1, 'ShowSettingsButton', ConfShowSettingsButton);
    Ini.WriteBool(Section1, 'ShowPlayButton', ConfShowPlayButton);
    Ini.WriteBool(Section1, 'ShowSearch', ConfShowSearch);
    Ini.WriteBool(Section1, 'Maximized', ConfMaximized);
    Ini.WriteInteger(Section1, 'Left', ConfWindowLeft);
    Ini.WriteInteger(Section1, 'Top', ConfWindowTop);
    Ini.WriteInteger(Section1, 'Width', ConfWindowWidth);
    Ini.WriteInteger(Section1, 'Height', ConfWindowHeight);
    Ini.WriteInteger(Section1, 'MovieListWidth', ConfMovieListWidth);

    Ini.WriteInteger(Section1, 'LastGroup', Ord(ConfGroupKind));
    Ini.WriteBool(Section1, 'LastGroupDesc', ConfGroupOrderDesc);
    Ini.WriteString(Section1, 'LastGroupCFTag', ConfGroupCFTag);
    Ini.WriteInteger(Section1, 'LastOrder', Ord(ConfOrderKind));
    Ini.WriteBool(Section1, 'LastOrderDesc', ConfOrderOrderDesc);
    Ini.WriteString(Section1, 'LastOrderCFTag', ConfOrderCFTag);

    // save template
    Ini.EraseSection(Section2);
    if (ConfTemplate.Count > 0) then
    begin
      Ini.WriteInteger(Section2, 'Count', ConfTemplate.Count);
      for x := 0 to ConfTemplate.Count-1
        do Ini.WriteString(Section2, 'Line' + IntToStr(x+1), ConfTemplate[x]);
    end;

    // save list template
    Ini.EraseSection(Section3);
    if (ConfListTemplate.Count > 0) then
    begin
      Ini.WriteInteger(Section3, 'Count', ConfListTemplate.Count);
      for x := 0 to ConfListTemplate.Count-1
        do Ini.WriteString(Section3, 'Line' + IntToStr(x+1), ConfListTemplate[x]);
    end;

    // save user vars
    Ini.EraseSection(Section4);
    if (ConfUserVars.Count > 0) then
    begin
      Ini.WriteInteger(Section4, 'Count', ConfUserVars.Count);
      for x := 0 to ConfUserVars.Count-1 do
      begin
        StrName := ConfUserVars.Names[x];
        StrValue := ConfUserVars.Values[StrName];
        if (StrName <> '') then
        begin
          Ini.WriteString(Section4, 'Name' + IntToStr(x+1), StrName);
          Ini.WriteString(Section4, 'Value' + IntToStr(x+1), StrValue);
        end;
      end;  
    end;

  finally
    Ini.Free;
  end;
end;

procedure TFormMain.ApplyConfig;
var
  x, y: Integer;
  Item: TSideBarItem;
begin
  if ConfFullScreen then
  begin
    BorderStyle := bsNone;
    WindowState := wsMaximized;
  end else
  begin
    BorderStyle := bsSizeable;
    WindowState := wsNormal;
    Left := ConfWindowLeft;
    Top := ConfWindowTop;
    Width := ConfWindowWidth;
    Height := ConfWindowHeight;
    Panel2.Width := ConfMovieListWidth;
  end;
  PanelClose.Visible := ConfFullScreen;
  LoadScheme(ConfColorScheme);
  LoadLanguage(ConfLanguage);
  AlignButtons;
  if ConfShowTopImages and (not GalleryPrepared)
    then PrepareGallery;
  NiceGallery1.Visible := ConfShowTopImages;
  NiceGallery1.Active := ConfShowTopImages and ConfAutoscroll;
  OpenItems(CurrentItem);
  if (ConfTitleType <> OldTitleType) then
  begin
    if GalleryPrepared then
    begin
      for x := 0 to NiceGallery1.Items.Count-1
        do NiceGallery1.Items[x].Hint := GetMovieTitle(TMovie(NiceGallery1.Items[x].Tag));
    end;
    for x := 0 to NiceSideBar1.Items.Count-1 do
    begin
      Item := NiceSideBar1.Items[x];
      for y := 0 to Item.Items.Count-1
        do Item.Items[y] := GetMovieTitle(TMovie(Item.Items.Objects[y]));
    end;
  end;
end;

procedure TFormMain.LoadScheme(Section: string);
const
  CRLF = #13#10;
var
  Ini: TIniFile;
  s1, s2, s3, s4, s5, s6, s7, s8: string;

  function ReadColor(Ident, Default: string): TColor;
  var
    Str: string;
  begin
    Str := Trim(Ini.ReadString(Section, Ident, Default));
    Result := StrToIntDef('$' + Copy(Str, 6, 2) + Copy(Str, 4, 2) + Copy(Str, 2, 2), 0);
  end;

begin
  Ini := TIniFile.Create(ExtractFilePath(Application.ExeName) + 'Schemes.ini');
  try

    NiceGallery1.Color := ReadColor('GalleryColor', '#808080');
    Panel1.Color := ReadColor('TopPanelColor', '#808080');
    Label1.Font.Color := ReadColor('Title1Color', '#000000');
    Label2.Font.Color := ReadColor('Title2Color', '#FFFFFF');

    BtnOpen.Color := ReadColor('ButtonColor', '#808080');
    BtnOpen.ColorOver := ReadColor('ButtonHoverColor', '#AAAAAA');
    BtnOpen.ColorDown := ReadColor('ButtonClickColor', '#AAAAAA');
    BtnOpen.Font.Color := ReadColor('ButtonTextColor', '#FFFFFF');
    BtnOpen.ColorOverCaption := ReadColor('ButtonTextHoverColor', '#FFFFFF');
    BtnOpen.ColorBorder := ReadColor('ButtonBorderColor', '#C0C0C0');

    BtnEdit.Color := BtnOpen.Color;
    BtnEdit.ColorOver := Btnopen.ColorOver;
    BtnEdit.ColorDown := BtnOpen.ColorDown;
    BtnEdit.Font.Color := BtnOpen.Font.Color;
    BtnEdit.ColorOverCaption := BtnOpen.ColorOverCaption;
    BtnEdit.ColorBorder := BtnOpen.ColorBorder;

    BtnSettings.Color := BtnOpen.Color;
    BtnSettings.ColorOver := Btnopen.ColorOver;
    BtnSettings.ColorDown := BtnOpen.ColorDown;
    BtnSettings.Font.Color := BtnOpen.Font.Color;
    BtnSettings.ColorOverCaption := BtnOpen.ColorOverCaption;
    BtnSettings.ColorBorder := BtnOpen.ColorBorder;

    BtnPlay.Color := BtnOpen.Color;
    BtnPlay.ColorOver := Btnopen.ColorOver;
    BtnPlay.ColorDown := BtnOpen.ColorDown;
    BtnPlay.Font.Color := BtnOpen.Font.Color;
    BtnPlay.ColorOverCaption := BtnOpen.ColorOverCaption;
    BtnPlay.ColorBorder := BtnOpen.ColorBorder;

    BtnBack.Color := BtnOpen.Color;
    BtnBack.ColorOver := Btnopen.ColorOver;
    BtnBack.ColorDown := BtnOpen.ColorDown;
    BtnBack.Font.Color := BtnOpen.Font.Color;
    BtnBack.ColorOverCaption := BtnOpen.ColorOverCaption;
    BtnBack.ColorBorder := BtnOpen.ColorBorder;

    PanelSearch.Color := BtnOpen.Color;

    NiceSideBar1.Color := ReadColor('SideBarColor', '#808080');

    NiceSideBar1.ItemStyle.NormalColor := ReadColor('SideBarSectionColor', '#808080');
    NiceSideBar1.ItemStyle.HoverColor := ReadColor('SideBarSectionHoverColor', '#808080');
    NiceSideBar1.ItemStyle.SelectedColor := ReadColor('SideBarSectionSelectedColor', '$00AAAAAA');
    NiceSideBar1.ItemStyle.NormalFont.Color := ReadColor('SideBarSectionTextColor', '#FFFFFF');
    NiceSideBar1.ItemStyle.HoverFont.Color := ReadColor('SideBarSectionTextHoverColor', '#FFFF00');
    NiceSideBar1.ItemStyle.SelectedFont.Color := ReadColor('SideBarSectionTextSelectedColor', '#000000');
    NiceSideBar1.ItemStyle.LineColor := ReadColor('SideBarSectionLineColor', '#FFFFFF');

    NiceSideBar1.SubItemStyle.NormalColor := ReadColor('SideBarItemColor', '#808080');
    NiceSideBar1.SubItemStyle.HoverColor := ReadColor('SideBarItemHoverColor', '#808080');
    NiceSideBar1.SubItemStyle.SelectedColor := ReadColor('SideBarItemSelectedColor', '$00AAAAAA');
    NiceSideBar1.SubItemStyle.NormalFont.Color := ReadColor('SideBarItemTextColor', '#FFFFFF');
    NiceSideBar1.SubItemStyle.HoverFont.Color := ReadColor('SideBarItemTextHoverColor', '#FFFF00');
    NiceSideBar1.SubItemStyle.SelectedFont.Color := ReadColor('SideBarItemTextSelectedColor', '#000000');
    NiceSideBar1.SubItemStyle.LineColor := ReadColor('SideBarItemLineColor', '#C0C0C0');

    NiceSideBar1.Bullets.NormalColor := ReadColor('SideBarBulletColor', '#FFFFFF');
    NiceSideBar1.Bullets.NormalPenColor := NiceSideBar1.Bullets.NormalColor;
    NiceSideBar1.Bullets.HoverColor := ReadColor('SideBarBulletHoverColor', '#FFFF00');
    NiceSideBar1.Bullets.HoverPenColor := NiceSideBar1.Bullets.HoverColor;
    NiceSideBar1.Bullets.SelectedColor := ReadColor('SideBarBulletSelectedColor', '#000000');
    NiceSideBar1.Bullets.SelectedPenColor := NiceSideBar1.Bullets.SelectedColor;

    NiceSideBar1.Scrollers.NormalColor := ReadColor('SideBarScrollerColor', '#000000');
    NiceSideBar1.Scrollers.NormalArrowColor := ReadColor('SideBarScrollerArrowColor', '#FFFFFF');
    NiceSideBar1.Scrollers.HoverColor := ReadColor('SideBarScrollerHoverColor', '#FFFFFF');
    NiceSideBar1.Scrollers.HoverArrowColor := ReadColor('SideBarScrollerArrowHoverColor', '#000000');

    Splitter1.Color := ReadColor('SplitterColor', '#FFFFFF');
    
    s1 := Ini.ReadString(Section, 'ScrollBarArrowColor', '#000000'); 
    s2 := Ini.ReadString(Section, 'ScrollBar3DLightColor', '#AAAAAA');
    s3 := Ini.ReadString(Section, 'ScrollBarDarkShadowColor', '#000000');
    s4 := Ini.ReadString(Section, 'ScrollBarFaceColor', '#AAAAAA');
    s5 := Ini.ReadString(Section, 'ScrollBarHighlightColor', '#AAAAAA');
    s6 := Ini.ReadString(Section, 'ScrollBarShadowColor', '#AAAAAA');
    s7 := Ini.ReadString(Section, 'ScrollBarTrackColor', '#808080');
    s8 := Ini.ReadString(Section, 'PageBackgroundColor', '#FFFFFF');

    TemplateHeader :=
      '<html>' + CRLF +
      '<header>' + CRLF +
      '<style type="text/css">' + CRLF +
      '<!--' + CRLF +
      'body {' + CRLF +
      '  scrollbar-arrow-color: '      + s1 + ';' + CRLF +
      '  scrollbar-3dlight-color: '    + s2 + ';' + CRLF +
      '  scrollbar-darkshadow-color: ' + s3 + ';' + CRLF +
      '  scrollbar-face-color: '       + s4 + ';' + CRLF +
      '  scrollbar-highlight-color: '  + s5 + ';' + CRLF +
      '  scrollbar-shadow-color: '     + s6 + ';' + CRLF +
      '  scrollbar-track-color: '      + s7 + ';' + CRLF +
      '}' + CRLF +
      'p, td {' + CRLF +
      '  font-family: Verdana; font-size: 8pt;' + CRLF +
      '}' + CRLF +
      '-->' + CRLF +
      '</style>' + CRLF +
      '<body bgcolor="' + s8 + '" leftmargin="0" topmargin="0" marginwidth="0" marginheight="0">' + CRLF;
    
    TemplateFooter := '</body>' + CRLF + '</html>';

  finally
    Ini.Free;
  end;
end;

procedure TFormMain.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  if not CheckPassword
    then Action := caNone;
end;

function TFormMain.CheckPassword: Boolean;
begin
  if ConfProtected then
  begin
    FormPassword.ShowModal;
    if FormPassword.Done
      then Result := SameText(FormPassword.Edit1.Text, ConfPassword)
      else Result := False; 
  end else
    Result := True;
end;

procedure TFormMain.BtnCategClick(Sender: TObject);
var
  Pt: TPoint;
begin
  Pt := BtnCateg.ClientToScreen(Point(0, BtnCateg.Height));
  PopupMenu1.Popup(Pt.X, Pt.Y);
end;

procedure TFormMain.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key = VK_F10)
    then BtnSettingsClick(nil);

end;

procedure TFormMain.FormKeyPress(Sender: TObject; var Key: Char);
begin
//    if (Key = 'p') then
//      BtnPlayClick(nil);
end;

procedure TFormMain.AlignButtons;
const
  ButtonWidthBase = 40;
  ButtonMargin = 5;
var
  l: Integer;
begin
  PaintBox1.Canvas.Font.Assign(BtnOpen.Font);
  BtnOpen.Width := ButtonWidthBase + Canvas.TextWidth(BtnOpen.Caption);
  BtnEdit.Width := ButtonWidthBase + Canvas.TextWidth(BtnEdit.Caption);
  BtnSettings.Width := ButtonWidthBase + Canvas.TextWidth(BtnSettings.Caption);
  BtnPlay.Width := ButtonWidthBase + Canvas.TextWidth(BtnPlay.Caption);
  BtnBack.Width := ButtonWidthBase + Canvas.TextWidth(BtnBack.Caption);
  l := ButtonMargin;
  BtnOpen.Visible := ConfShowOpenButton;
  if ConfShowOpenButton then
  begin
    BtnOpen.Left := l;
    l := l + BtnOpen.Width + ButtonMargin;
  end;
  BtnEdit.Visible := ConfShowEditButton;
  if ConfShowEditButton then
  begin
    BtnEdit.Left := l;
    l := l + BtnEdit.Width + ButtonMargin;
  end;
  BtnSettings.Visible := ConfShowSettingsButton;
  if ConfShowSettingsButton then
  begin
    BtnSettings.Left := l;
    l := l + BtnSettings.Width + ButtonMargin;
  end;
  BtnPlay.Visible := ConfShowPlayButton;
  if ConfShowPlayButton then
  begin
    BtnPlay.Left := l;
    l := l + BtnPlay.Width + ButtonMargin;
  end;  
  PanelSearch.Visible := ConfShowSearch;
  if ConfShowSearch then
  begin
    PanelSearch.Left := l;
    l := BtnBack.Width + ButtonMargin;
    ComboSearch.Left := l;
    l := l + ComboSearch.Width + ButtonMargin;
    EditSearch.Left := l;
    l := l + EditSearch.Width + ButtonMargin;
    BtnSearch.Left := l;
    l := l + BtnSearch.Width + ButtonMargin;
    PanelSearch.Width := l;
  end;
end;

procedure TFormMain.GroupToInt(var Value: Integer; GroupKind: TSortKind; GroupCFTag: string);
begin
  if GroupKind = skCustomField then
  begin
    if List <> nil then
      Value := List.CustomFieldsProperties.IndexOf(GroupCFTag)
    else
      Value := -1;
    if Value = -1 then
      Value := Ord(0) // no group
    else
      Value := Value + CustomFieldStart;
  end else
  begin
    Value := Ord(GroupKind);
  end;
end;

procedure TFormMain.OrderToInt(var Value: Integer; SortKind: TSortKind; SortCFTag: string);
begin
  if SortKind = skCustomField then
  begin
    if List <> nil then
      Value := List.CustomFieldsProperties.IndexOf(SortCFTag)
    else
      Value := -1;
    if Value = -1 then
      Value := Ord(DefaultSortKind) // default sort kind
    else
      Value := Value + CustomFieldStart;
  end else
  begin
    Value := Ord(SortKind);
  end;
end;

procedure TFormMain.IntToGroup(Value: Integer; var GroupKind: TSortKind; var GroupCFTag: string);
begin
  if Value >= CustomFieldStart then
  begin
    if (List <> nil) and ((Value - CustomFieldStart) < List.CustomFieldsProperties.Count) then
    begin
      GroupCFTag := List.CustomFieldsProperties.Objects[Value - CustomFieldStart].FieldTag;
      GroupKind := skCustomField;
    end else
    begin
      GroupCFTag := '';
      GroupKind := skNone;
    end;
  end else
  begin
    GroupCFTag := '';
    GroupKind := TSortKind(Value);
  end;
end;

procedure TFormMain.IntToOrder(Value: Integer; var SortKind: TSortKind; var SortCFTag: string);
begin
  if Value >= CustomFieldStart then
  begin
    if (List <> nil) and ((Value - CustomFieldStart) < List.CustomFieldsProperties.Count) then
    begin
      SortCFTag := List.CustomFieldsProperties.Objects[Value - CustomFieldStart].FieldTag;
      SortKind := skCustomField;
    end else
    begin
      SortCFTag := '';
      SortKind := DefaultSortKind;
    end;
  end else
  begin
    SortCFTag := '';
    SortKind := TSortKind(Value);
  end;
end;

procedure TFormMain.SetupGroupMenus;

  procedure SetupMenu1(AMenu: TMenuItem; AKind: TSortKind);
  begin
    AMenu.Tag := Ord(AKind);
    AMenu.OnClick := GroupClick1;
  end;

  procedure SetupMenu2(AMenu: TMenuItem; AKind: TSortKind);
  begin
    AMenu.Tag := Ord(AKind);
    AMenu.OnClick := GroupClick2;
  end;

begin
  SetupMenu1(None1, skNone);
  SetupMenu1(Category1, skCategory);
  SetupMenu1(OriginalTitle1, skOrigTitle);
  SetupMenu1(TranslatedTitle1, skTransTitle);
  SetupMenu1(Rating1, skRating);
  SetupMenu1(Year1, skYear);
  SetupMenu1(Decade1, skDecade);
  SetupMenu1(Director1, skDirector);
  SetupMenu1(Producer1, skProducer);
  SetupMenu1(Actors1, skActor);
  SetupMenu1(Country1, skCountry);
  SetupMenu1(MovieNumber1, skNumber);
  SetupMenu1(DateAdded1, skDate);
  SetupMenu1(Media1, skMedia);
  SetupMenu1(MediaType1, skMediaType);
  SetupMenu1(Borrower1, skBorrower);
  SetupMenu1(Length1, skLength);
  SetupMenu1(VideoFormat1, skVideoFormat);
  SetupMenu1(VideoBitrate1, skVideoBitrate);
  SetupMenu1(AudioFormat1, skAudioFormat);
  SetupMenu1(AudioBitrate1, skAudioBitrate);
  SetupMenu1(Resolution1, skResolution);
  SetupMenu1(Framerate1, skFramerate);
  SetupMenu1(Disks1, skDisk);
  SetupMenu1(Languages1, skLanguage);
  SetupMenu1(Subtitles1, skSubtitle);

  SetupMenu2(Category2, skCategory);
  SetupMenu2(OriginalTitle2, skOrigTitle);
  SetupMenu2(TranslatedTitle2, skTransTitle);
  SetupMenu2(Rating2, skRating);
  SetupMenu2(Year2, skYear);
  SetupMenu2(Decade2, skDecade);
  SetupMenu2(Director2, skDirector);
  SetupMenu2(Producer2, skProducer);
  SetupMenu2(Actors2, skActor);
  SetupMenu2(Country2, skCountry);
  SetupMenu2(MovieNumber2, skNumber);
  SetupMenu2(DateAdded2, skDate);
  SetupMenu2(Media2, skMedia);
  SetupMenu2(MediaType2, skMediaType);
  SetupMenu2(Borrower2, skBorrower);
  SetupMenu2(Length2, skLength);
  SetupMenu2(VideoFormat2, skVideoFormat);
  SetupMenu2(VideoBitrate2, skVideoBitrate);
  SetupMenu2(AudioFormat2, skAudioFormat);
  SetupMenu2(AudioBitrate2, skAudioBitrate);
  SetupMenu2(Resolution2, skResolution);
  SetupMenu2(Framerate2, skFramerate);
  SetupMenu2(Disks2, skDisk);
  SetupMenu2(Languages2, skLanguage);
  SetupMenu2(Subtitles2, skSubtitle);

  Ascending1.Tag := 0;
  Ascending1.OnClick := AscClick1;
  Descending1.Tag := 1;
  Descending1.OnClick := AscClick1;

  Ascending2.Tag := 0;
  Ascending2.OnClick := AscClick2;
  Descending2.Tag := 1;
  Descending2.OnClick := AscClick2;
end;

procedure TFormMain.FillGroupMenu;
var
  newMenuItem: TMenuItem;
  i: Integer;
begin
  with Others1 do
    while Count > NbFixedGroupItems do
      Delete(Count-1);

  if (List <> nil) then
    with List.CustomFieldsProperties do
    begin
      for i := 0 to Count-1 do
      begin
        newMenuItem := TMenuItem.Create(Others1);
        Others1.Add(newMenuItem);
        with newMenuItem do
        begin
          Caption := Objects[i].FieldName;
          Tag := CustomFieldStart + i;
          OnClick := GroupClick1;
        end;
      end;
    end;
end;

procedure TFormMain.FillSortMenu;
var
  newMenuItem: TMenuItem;
  i: Integer;
begin
  with Others2 do
    while Count > NbFixedSortItems do
      Delete(Count-1);

  if (List <> nil) then
    with List.CustomFieldsProperties do
    begin
      for i := 0 to Count-1 do
      begin
        newMenuItem := TMenuItem.Create(Others2);
        Others2.Add(newMenuItem);
        with newMenuItem do
        begin
          Caption := Objects[i].FieldName;
          Tag := CustomFieldStart + i;
          OnClick := GroupClick2;
        end;
      end;
    end;
end;

procedure TFormMain.CheckGroup1;
var
  Tg, i: Integer;
begin
  GroupToInt(Tg, ConfGroupKind, ConfGroupCFTag);
  None1.Checked := None1.Tag = Tg;
  Category1.Checked := Category1.Tag = Tg;
  OriginalTitle1.Checked := OriginalTitle1.Tag = Tg;
  TranslatedTitle1.Checked := TranslatedTitle1.Tag = Tg;
  Rating1.Checked := Rating1.Tag = Tg;
  Year1.Checked := Year1.Tag = Tg;
  Decade1.Checked := Decade1.Tag = Tg;
  Director1.Checked := Director1.Tag = Tg;
  Producer1.Checked := Producer1.Tag = Tg;
  Actors1.Checked := Actors1.Tag = Tg;
  Country1.Checked := Country1.Tag = Tg;
  MovieNumber1.Checked := MovieNumber1.Tag = Tg;
  DateAdded1.Checked := DateAdded1.Tag = Tg;
  Media1.Checked := Media1.Tag = Tg;
  MediaType1.Checked := MediaType1.Tag = Tg;
  Borrower1.Checked := Borrower1.Tag = Tg;
  Length1.Checked := Length1.Tag = Tg;
  VideoFormat1.Checked := VideoFormat1.Tag = Tg;
  VideoBitrate1.Checked := VideoBitrate1.Tag = Tg;
  AudioFormat1.Checked := AudioFormat1.Tag = Tg;
  AudioBitrate1.Checked := AudioBitrate1.Tag = Tg;
  Resolution1.Checked := Resolution1.Tag = Tg;
  Framerate1.Checked := Framerate1.Tag = Tg;
  Disks1.Checked := Disks1.Tag = Tg;
  Languages1.Checked := Languages1.Tag = Tg;
  Subtitles1.Checked := Subtitles1.Tag = Tg;
  for i := NbFixedGroupItems to Others1.Count-1 do
    Others1.Items[i].Checked := Others1.Items[i].Tag = Tg;
end;

procedure TFormMain.CheckGroup2;
var
  Tg, i: Integer;
begin
  OrderToInt(Tg, ConfOrderKind, ConfOrderCFTag);
  Category2.Checked := Category2.Tag = Tg;
  OriginalTitle2.Checked := OriginalTitle2.Tag = Tg;
  TranslatedTitle2.Checked := TranslatedTitle2.Tag = Tg;
  Rating2.Checked := Rating2.Tag = Tg;
  Year2.Checked := Year2.Tag = Tg;
  Decade2.Checked := Decade2.Tag = Tg;
  Director2.Checked := Director2.Tag = Tg;
  Producer2.Checked := Producer2.Tag = Tg;
  Actors2.Checked := Actors2.Tag = Tg;
  Country2.Checked := Country2.Tag = Tg;
  MovieNumber2.Checked := MovieNumber2.Tag = Tg;
  DateAdded2.Checked := DateAdded2.Tag = Tg;
  Media2.Checked := Media2.Tag = Tg;
  MediaType2.Checked := MediaType2.Tag = Tg;
  Borrower2.Checked := Borrower2.Tag = Tg;
  Length2.Checked := Length2.Tag = Tg;
  VideoFormat2.Checked := VideoFormat2.Tag = Tg;
  VideoBitrate2.Checked := VideoBitrate2.Tag = Tg;
  AudioFormat2.Checked := AudioFormat2.Tag = Tg;
  AudioBitrate2.Checked := AudioBitrate2.Tag = Tg;
  Resolution2.Checked := Resolution2.Tag = Tg;
  Framerate2.Checked := Framerate2.Tag = Tg;
  Disks2.Checked := Disks2.Tag = Tg;
  Languages2.Checked := Languages2.Tag = Tg;
  Subtitles2.Checked := Subtitles2.Tag = Tg;
  for i := NbFixedGroupItems to Others2.Count-1 do
    Others2.Items[i].Checked := Others2.Items[i].Tag = Tg;
end;

procedure TFormMain.CheckAsc1;
begin
  Ascending1.Checked := not ConfGroupOrderDesc;
  Descending1.Checked := ConfGroupOrderDesc;
end;

procedure TFormMain.CheckAsc2;
begin
  Ascending2.Checked := not ConfOrderOrderDesc;
  Descending2.Checked := ConfOrderOrderDesc;
end;

procedure TFormMain.GroupClick1(Sender: TObject);
begin
  IntToGroup((Sender as TMenuItem).Tag, ConfGroupKind, ConfGroupCFTag);
  CheckGroup1;
  GroupMovies;
end;

procedure TFormMain.GroupClick2(Sender: TObject);
begin
  IntToOrder((Sender as TMenuItem).Tag, ConfOrderKind, ConfOrderCFTag);
  CheckGroup2;
  GroupMovies;
end;

procedure TFormMain.AscClick1(Sender: TObject);
begin
  ConfGroupOrderDesc := (Sender as TMenuItem).Tag = 1;
  CheckAsc1;
  GroupMovies;
end;

procedure TFormMain.AscClick2(Sender: TObject);
begin
  ConfOrderOrderDesc := (Sender as TMenuItem).Tag = 1;
  CheckAsc2;
  GroupMovies;
end;

procedure TFormMain.FillComboSearch;
var
  lastItemIndex, i: Integer;
begin
  lastItemIndex := ComboSearch.ItemIndex;
  ComboSearch.Items.Clear;
  ComboSearch.Items.Add(_('Title'));
  ComboSearch.Items.Add(_('Original Title'));
  ComboSearch.Items.Add(_('Translated Title'));
  ComboSearch.Items.Add(_('Actors'));
  ComboSearch.Items.Add(_('Director'));
  ComboSearch.Items.Add(_('Producer'));
  ComboSearch.Items.Add(_('Description'));
  ComboSearch.Items.Add(_('Comments'));
  ComboSearch.Items.Add(_('Category'));
  ComboSearch.Items.Add(_('Country'));
  ComboSearch.Items.Add(_('Borrower'));
  ComboSearch.Items.Add(_('Media'));
  ComboSearch.Items.Add(_('Media Type'));
  ComboSearch.Items.Add(_('Source'));
  ComboSearch.Items.Add(_('URL'));
  ComboSearch.Items.Add(_('Movie Number'));
  ComboSearch.Items.Add(_('Date Added'));
  ComboSearch.Items.Add(_('Rating'));
  ComboSearch.Items.Add(_('Year'));
  ComboSearch.Items.Add(_('Length'));
  ComboSearch.Items.Add(_('Resolution'));
  ComboSearch.Items.Add(_('Video Format'));
  ComboSearch.Items.Add(_('Video Bitrate'));
  ComboSearch.Items.Add(_('Audio Format'));
  ComboSearch.Items.Add(_('Audio Bitrate'));
  ComboSearch.Items.Add(_('Size'));
  ComboSearch.Items.Add(_('Framerate'));
  ComboSearch.Items.Add(_('Disks'));
  ComboSearch.Items.Add(_('Languages'));
  ComboSearch.Items.Add(_('Subtitles'));
  //Custom fields
  if List <> nil then
    for i := 0 to List.CustomFieldsProperties.Count-1 do
      ComboSearch.Items.Add(List.CustomFieldsProperties.Objects[i].FieldName);
  if (lastItemIndex >= 0) and (lastItemIndex < ComboSearch.Items.Count) then
    ComboSearch.ItemIndex := lastItemIndex
  else
    ComboSearch.ItemIndex := 0;
end;

//Alex:allow searches
{
http://www.mindspring.com/~cityzoo/tips/srchstr.txt

From: stidolph@magnet.com (David Stidolph)
Subject: [delphi] String Pattern matching
Date: Tue, 27 Jun 1995 10:01:18 -0400

There are many times when you need to compare two strings, but want to use
wild cards in the match - all last names that begin with 'St', etc.  The
following is a piece of code I got from Sean Stanley in Tallahassee Florida
in C.  I translated it into Delphi an am uploading it here for all to use.
I have not tested it extensivly, but the original function has been tested
quite thoughly.

I would love feedback on this routine - or peoples changes to it.  I want to
forward them to Sean to get him to release more tidbits like this.
}
{
  This function takes two strings and compares them.  The first string
  can be anything, but should not contain pattern characters (* or ?).
  The pattern string can have as many of these pattern characters as you want.
  For example: MatchStrings('David Stidolph','*St*') would return True.

  Orignal code by Sean Stanley in C
  Rewritten in Delphi by David Stidolph
}
function MatchStrings(Source, Pattern: string): Boolean;

  function MatchPattern(Element, Pattern: PChar): Boolean;

    function IsPatternWild(Pattern: PChar): Boolean;
    begin
      Result := StrScan(Pattern,'*') <> nil;
      if not Result
        then Result := StrScan(Pattern,'?') <> nil;
    end;

  begin
    if (StrComp(Pattern,'*') = 0)
      then Result := True else
    if (Element^ = Chr(0)) and (Pattern^ <> Chr(0))
      then Result := False else
    if Element^ = Chr(0)
      then Result := True
      else
      begin
        case Pattern^ of
          '*': if MatchPattern(Element, @Pattern[1])
                 then Result := True
                 else Result := MatchPattern(@Element[1], Pattern);
          '?': Result := MatchPattern(@Element[1], @Pattern[1]);
        else
          if Element^ = Pattern^
            then Result := MatchPattern(@Element[1], @Pattern[1])
            else Result := False;
        end;
      end;
  end;

begin
  Result := MatchPattern(PChar(Source), PChar(Pattern));
end;

procedure TFormMain.Find(Term: string);
var
  i, NbFound: Integer;
  Movie: TMovie;
  Str, mt: string;
  Movies: string;

  function AddWithComma(SomeStr, Addition: string): string;
  begin
    if (SomeStr = '')
      then Result := Addition
      else Result := SomeStr + ',' + Addition;
  end;
  
  function IntToStrValue(i: Integer): string;
  begin
    if i = -1 then
      Result := ''
    else
      Result := IntToStr(i);
  end;
  
  function FloatToStrValue(r: Extended): string;
  begin
    if r < 0 then
      Result := ''
    else
      Result := FormatFloat('#0.0', r, FormatSettings);
  end;
  
  function DateToStrValue(d: Integer): string;
  begin
    if d = -1 then
      Result := ''
    else
      Result := FormatDateTime(ConfDateFormat, d)
  end;

begin

  if (Term = '')
    then Exit;

  Str := Term;
  if (Copy(Term, 1, 1) <> '"') then
  begin
    Str := '*' + Term + '*';
    Str := StringReplace(Str, ' ', '*', [rfReplaceAll]);
  end;

  Str := StringReplace(Str, '"', '', [rfReplaceAll]);
  Str := LowerCase(Str);

  Movies := '';
  NbFound := 0;
  for i := 0 to List.Count-1 do
  begin
    if NbFound >= 40 then
    begin
      Movies := Movies + ',...';
      Break;
    end;
    Movie := TMovie(List[i]);
    mt := '';
    
    case ComboSearch.ItemIndex of
      0: mt := GetMovieTitle(Movie);                // Title
      1: mt := Movie.strOriginalTitle;              // Original Title
      2: mt := Movie.strTranslatedTitle;            // Translated Title
      3: mt := Movie.strActors;                     // Actors
      4: mt := Movie.strDirector;                   // Director
      5: mt := Movie.strProducer;                   // Producer
      6: mt := Movie.strDescription;                // Description
      7: mt := Movie.strComments;                   // Comments
      8: mt := Movie.strCategory;                   // Category
      9: mt := Movie.strCountry;                    // Country
     10: mt := Movie.strBorrower;                   // Borrower
     11: mt := Movie.strMedia;                      // Media
     12: mt := Movie.strMediaType;                  // Media Type
     13: mt := Movie.strSource;                     // Source
     14: mt := Movie.strURL;                        // URL
     15: mt := IntToStrValue(Movie.iNumber);        // Movie Number
     16: mt := DateToStrValue(Movie.iDate);         // Date Added
     17: mt := FloatToStrValue(Movie.iRating/10);   // Rating
     18: mt := IntToStrValue(Movie.iYear);          // Year
     19: mt := IntToStrValue(Movie.iLength);        // Length
     20: mt := Movie.strResolution;                 // Resolution
     21: mt := Movie.strVideoFormat;                // Video Format
     22: mt := IntToStrValue(Movie.iVideoBitrate);  // Video Bitrate
     23: mt := Movie.strAudioFormat;                // Audio Format
     24: mt := IntToStrValue(Movie.iAudioBitrate);  // Audio Bitrate
     25: mt := Movie.strSize;                       // Size
     26: mt := Movie.strFramerate;                  // Framerate
     27: mt := IntToStrValue(Movie.iDisks);         // Disks
     28: mt := Movie.strLanguages;                  // Languages
     29: mt := Movie.strSubtitles;                  // Subtitles
     else
      begin
        //Custom fields
        mt := Movie.CustomFields.GetFieldValue(List.CustomFieldsProperties[ComboSearch.ItemIndex-30].FieldTag, False, False);
        if (mt <> '') and (List.CustomFieldsProperties[ComboSearch.ItemIndex-30].FieldType = ftDate) then
          mt := FormatDateTime(ConfDateFormat, StrToDate(mt, FormatSettings));
      end;
    end;
    if (mt = '')
      then Continue;
    mt := LowerCase(mt);
    //remove wildcards
    mt := StringReplace(mt, '*', '', [rfReplaceAll]);
    mt := StringReplace(mt, '?', '', [rfReplaceAll]);
    if (mt = '')
      then Continue;
    if MatchStrings(mt, Str) then
    begin
      Movies := AddWithComma(Movies, IntToStr(Integer(Movie)));
      Inc(NbFound);
    end;
  end;

  if (Movies = '') then
  begin
    Movies := StringReplace(Term, '"', '', [rfReplaceAll]);
    if (Movies = '') then
      Movies := '"empty"'
    else
      Movies := '"' + Movies + '"';
  end;

  OpenItems(Movies);
end;

// Alex
procedure TFormMain.EditSearchKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_RETURN) and (EditSearch.Text <> '')
    then Find(EditSearch.Text);
end;

procedure TFormMain.EditSearchClick(Sender: TObject);
begin
   EditSearch.SelectAll;
end;

procedure TFormMain.BtnSearchClick(Sender: TObject);
begin
  if (EditSearch.Text <> '')
    then Find(EditSearch.Text);
end;

procedure TFormMain.BtnCloseClick(Sender: TObject);
begin
  Close;
end;

procedure TFormMain.LoadLanguage(Language: string);
begin
  TransLoad(ExtractFilePath(Application.ExeName) + ConfLanguage + '.lng');
  UpdateLanguage;
  FormSettings.UpdateLanguage;
end;

procedure TFormMain.UpdateLanguage;
begin
  BtnClose.Caption := _('Close');
  BtnOpen.Caption := ' ' + _('Open');
  BtnEdit.Caption := ' ' + _('Edit');
  BtnSettings.Caption := ' ' + _('Settings');
  BtnPlay.Caption := ' ' + _('Play');
  BtnBack.Caption := ' ' + _('Back');
  BtnEdit.Caption := ' ' + _('Edit');

  if (List.MovieProperties.strName <> '')
    then Label1.Caption := List.MovieProperties.strName + '''s'
    else Label1.Caption := _('My Private');

  Label2.Caption := _('Movie Collection');

  case List.Count of
    0: Label4.Caption := _('No Movie');
    1: Label4.Caption := _('1 Movie');
  else
    Label4.Caption := IntToStr(List.Count) + ' ' + _('Movies');
  end;

  if (List.Count > 0)
    then Label5.Caption := ConfCatalogFile
    else Label5.Caption := _('Open a catalog file first ...');

  FillComboSearch;
  ComboSearch.ItemIndex := 0;

  GroupMovies;

  Groupby1.Caption := _('Group by');
  None1.Caption := _('<none>');
  Category1.Caption := _('Category');
  OriginalTitle1.Caption := _('Original Title');
  TranslatedTitle1.Caption := _('Translated Title');
  Rating1.Caption := _('Rating');
  Year1.Caption := _('Year');
  Decade1.Caption := _('Decade');
  Director1.Caption := _('Director');
  Producer1.Caption := _('Producer');
  Actors1.Caption := _('Actors');
  Country1.Caption := _('Country');
  MovieNumber1.Caption := _('Movie Number');
  DateAdded1.Caption := _('Date Added');
  Media1.Caption := _('Media');
  MediaType1.Caption := _('Media Type');
  Borrower1.Caption := _('Borrower');
  Length1.Caption := _('Length');
  VideoFormat1.Caption := _('Video Format');
  VideoBitrate1.Caption := _('Video Bitrate');
  AudioFormat1.Caption := _('Audio Format');
  AudioBitrate1.Caption := _('Audio Bitrate');
  Resolution1.Caption := _('Resolution');
  Framerate1.Caption := _('Framerate');
  Disks1.Caption := _('Disks');
  Languages1.Caption := _('Languages');
  Subtitles1.Caption := _('Subtitles');
  Others1.Caption := _('Others');
  Ascending1.Caption := _('Ascending');
  Descending1.Caption := _('Descending');

  Orderby1.Caption := _('Order by');
  Category2.Caption := _('Category');
  OriginalTitle2.Caption := _('Original Title');
  TranslatedTitle2.Caption := _('Translated Title');
  Rating2.Caption := _('Rating');
  Year2.Caption := _('Year');
  Decade2.Caption := _('Decade');
  Director2.Caption := _('Director');
  Producer2.Caption := _('Producer');
  Actors2.Caption := _('Actors');
  Country2.Caption := _('Country');                                          
  MovieNumber2.Caption := _('Movie Number');
  DateAdded2.Caption := _('Date Added');
  Media2.Caption := _('Media');
  MediaType2.Caption := _('Media Type');
  Borrower2.Caption := _('Borrower');
  Length2.Caption := _('Length');
  VideoFormat2.Caption := _('Video Format');
  VideoBitrate2.Caption := _('Video Bitrate');
  AudioFormat2.Caption := _('Audio Format');
  AudioBitrate2.Caption := _('Audio Bitrate');
  Resolution2.Caption := _('Resolution');
  Framerate2.Caption := _('Framerate');                                       
  Disks2.Caption := _('Disks');
  Languages2.Caption := _('Languages');
  Subtitles2.Caption := _('Subtitles');
  Others2.Caption := _('Others');
  Ascending2.Caption := _('Ascending');
  Descending2.Caption := _('Descending');
                                                                     
end;

procedure TFormMain.WebBrowser1CommandStateChange(Sender: TObject;
  Command: Integer; Enable: WordBool);
begin
  case Command of
    CSC_NAVIGATEBACK: BtnBack.Enabled := Enable;
  end;
end;

procedure TFormMain.ClearAllHistory();
begin
  ClearHistory := True;
  while BtnBack.Enabled do
  begin
    WebBrowser1.GoBack;
  end;
  ClearHistory := False;
end;

procedure TFormMain.WebBrowser1BeforeNavigate2(Sender: TObject;
  const pDisp: IDispatch; var URL, Flags, TargetFrameName, PostData,
  Headers: OleVariant; var Cancel: WordBool);
begin
  ClearImages;
end;

procedure TFormMain.Panel2Resize(Sender: TObject);
begin
  // Change position of group/sort button
  BtnCateg.Left := NiceSideBar1.Width-17;
  // Rebuild items to change their size
  NiceSideBar1.BeginUpdate;
  NiceSideBar1.EndUpdate;
end;

initialization

  OleInitialize(nil);

finalization

  OleUninitialize;

end.
