{-------------------------------------------------------------------------------

The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is "TransUtils.pas" released at July 4th, 2008.

The Initial Developer of the Original Code is

  Priyatna
  Website: http://www.priyatna.org
  Email: me@priyatna.org
  Copyright (c) 2008 Priyatna
  All Rights Reserved.

Contributor(s): -

Description: A simple translation engine.

Known Issues:
  - Only supports ASCII.
  - Doesn't support '=' properly in keys. TIniFile limitation.
  - Case-sensitive by design. Is TIniFile case-sensitive?

Last Modified: 2008-07-24

-------------------------------------------------------------------------------}

unit TransUtils;

interface

uses
  Windows, Classes, SysUtils;

  function _(const Str: string): string;

  procedure TransLoad(FileName: string);

  procedure TransSave(FileName: string);

implementation

uses
  IniFiles;

const
  BUCKET_COUNT = 100;

type
  PTransItem = ^TTransItem;
  TTransItem = record
    Original: string;
    Translation: string;
    Next: PTransItem;
  end;

var
  Inited: Boolean = False;
  HashTable: array [0..BUCKET_COUNT-1] of PTransItem;

function HashString(const Str: string): Integer;
var
  i: Integer;
begin
  Result := 0;
  for i := 1 to Length(Str)
    do Result := Result + Ord(Str[i]);
  Result := Result mod BUCKET_COUNT;
end;  

function _(const Str: string): string;
var
  P: PTransItem;
begin
  if Inited and (Str <> '') then
  begin
    P := HashTable[HashString(Str)];
    if Assigned(P) then
    begin
      while Assigned(P) do
      begin
        if (P^.Original = Str) then
        begin
          Result := P^.Translation;
          Exit;
        end;
        P := P^.Next;
      end;
    end;
  end;
  Result := Str;
end;

procedure TransClear;
var
  x: Integer;

  procedure FreeItems(First: PTransItem);
  var
    P, Next: PTransItem;
  begin
    P := First;
    while Assigned(P) do
    begin
      Next := P^.Next;
      P^.Original := '';
      P^.Translation := '';
      Dispose(P);
      P := Next;
    end;
  end;

begin
  for x := 0 to BUCKET_COUNT-1 do
  begin
    FreeItems(HashTable[x]);
    HashTable[x] := nil;
  end;
  Inited := False;
end;

procedure TransAdd(const Original, Translation: string);
var
  Hash: Integer;
  P, Last: PTransItem;
begin
  if (Original = '') or (Translation = '')
    then Exit;
  Hash := HashString(Original);
  P := HashTable[Hash];
  Last := P;
  if Assigned(P) then
  begin
    while Assigned(P) do
    begin
      if (P^.Original = Original) then
      begin
        P^.Translation := Translation;
        Exit;
      end;
      Last := P;
      P := Last^.Next;
    end;
  end;
  P := New(PTransItem);
  P^.Original := Original;
  P^.Translation := Translation;
  P^.Next := nil;
  if Assigned(Last)
    then Last^.Next := P
    else HashTable[Hash] := P;
end;

procedure TransLoad(FileName: string);
const
  Section = 'Translations';
var
  Ini: TIniFile;
  t: TStringList;
  x: Integer;
begin
  TransClear;
  if not FileExists(FileName)
    then Exit;
  Ini := TIniFile.Create(FileName);
  t := TStringList.Create;
  try
    Ini.ReadSection(Section, t);
    for x := 0 to t.Count-1
      do TransAdd(t[x], Ini.ReadString(Section, t[x], ''));
  finally
    t.Free;
    Ini.Free;
  end;
  Inited := True;
end;

procedure TransSave(FileName: string);
const
  Section = 'Translations';
var
  Ini: TIniFile;
  P: PTransItem;
  x: Integer;
begin
  Ini := TIniFile.Create(FileName);
  try
    for x := 0 to BUCKET_COUNT-1 do
    begin
      P := HashTable[x];
      while Assigned(P) do
      begin
        Ini.WriteString(Section, P^.Original, P^.Translation);
        P := P^.Next;
      end;
    end;
  finally
    Ini.Free;
  end;
end;

end.
