{-------------------------------------------------------------------------------

The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is "GroupSortUtils.pas", released at January 12nd, 2008.

The Initial Developer of the Original Code is

  Priyatna
  Website: http://www.priyatna.org
  Email: me@priyatna.org
  Copyright (c) 2008 Priyatna
  All Rights Reserved.
        
Contributor(s): -

Description: Group and sort movies.

Known Issues: -

Last Modified: 2008-07-24

-------------------------------------------------------------------------------}

unit GroupSortUtils;

interface

uses
  Windows, Classes, SysUtils, MovieClass, NiceSideBar, DateUtils, Dialogs;

type
  TSortKind = (
    skNone, skCategory, skOrigTitle, skTransTitle, skRating, skYear, skDecade,
    skDirector, skProducer, skCountry, skLanguage, skNumber, skMedia, skMediaType,
    skDate, skBorrower, skLength, skActor, skVideoFormat, skVideoBitrate,
    skAudioFormat, skAudioBitrate, skResolution, skFramerate, skSubtitle,
    skDisk
  );

const
  DefaultSortKind: TSortKind = skNumber;

  procedure GroupAndSortMovies(List: TList; Bar: TNiceSideBar; TitleType: Integer;
    GroupKind: TSortKind; GroupReverseOrder: Boolean;
    SortKind: TSortKind; SortReverseOrder: Boolean;
    var Title: string);


implementation

uses
  TransUtils;

//------------------------------------------------------------------------------

function SortIntAscd(List: TStringList; Index1, Index2: Integer): Integer;
var
  Int1, Int2: Integer;
begin
  Int1 := StrToIntDef(List[Index1], 0);
  Int2 := StrToIntDef(List[Index2], 0);
  if (Int1 = Int2)
    then Result := 0 else
  if (Int1 > Int2)
    then Result := 1
    else Result := -1;
end;

function SortIntDscd(List: TStringList; Index1, Index2: Integer): Integer;
var
  Int1, Int2: Integer;
begin
  Int1 := StrToIntDef(List[Index1], 0);
  Int2 := StrToIntDef(List[Index2], 0);
  if (Int1 = Int2)
    then Result := 0 else
  if (Int1 > Int2)
    then Result := -1
    else Result := 1;
end;

function SortStrAscd(List: TStringList; Index1, Index2: Integer): Integer;
begin
  Result := CompareText(List[Index1], List[Index2]);
end;

function SortStrDscd(List: TStringList; Index1, Index2: Integer): Integer;
begin
  Result := -CompareText(List[Index1], List[Index2]);
end;

//------------------------------------------------------------------------------

procedure FlatCommaText(List: TStringList; const Txt: string; Movie: TMovie);
var
  Last, Str: string;
  i, l: Integer;
  c: Char;
  ParenCount: Integer;
begin
  if (Txt = '') then
  begin
    List.AddObject(_('Others'), Movie);
    Exit;
  end;
  Str := '';
  Last := '';
  l := Length(Txt);
  i := 1;
  while (i <= l) do
  begin
    c := Txt[i];
    Inc(i);
    case c of
      ',': begin
             Str := Trim(Str);
             if (Str <> '') and (Last <> Str)
               then List.AddObject(Str, Movie);
             Last := Str;
             Str := '';
           end;
      '(': begin
             ParenCount := 1;
             while (i <= l) do
             begin
               c := Txt[i];
               Inc(i);
               case c of
                 '(': ParenCount := ParenCount + 1;
                 ')': ParenCount := ParenCount - 1;
               end;
               if (ParenCount = 0)
                 then Break;
             end;
           end;
    else
      Str := Str + c;
    end;
  end;
  Str := Trim(Str);
  if (Str <> '') and (Last <> Str)
    then List.AddObject(Str, Movie);
end;

procedure FlatCategory(List: TStringList; Movie: TMovie);
begin
  FlatCommaText(List, Movie.strCategory, Movie);
end;

procedure FlatOriginalTitle(List: TStringList; Movie: TMovie);
begin
  List.AddObject(UpperCase(Copy(Movie.strOriginalTitle, 1, 1)), Movie);
end;

procedure FlatTranslatedTitle(List: TStringList; Movie: TMovie);
begin
  if (Movie.strTranslatedTitle <> '')
    then List.AddObject(UpperCase(Copy(Movie.strTranslatedTitle, 1, 1)), Movie)
    else List.AddObject(UpperCase(Copy(Movie.strOriginalTitle, 1, 1)), Movie);
end;

procedure FlatRating(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iRating), Movie);
end;

procedure FlatYear(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iYear), Movie);
end;

procedure FlatDecade(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iYear), Movie);
end;

procedure FlatDirector(List: TStringList; Movie: TMovie);
begin
  FlatCommaText(List, Movie.strDirector, Movie);
end;

procedure FlatProducer(List: TStringList; Movie: TMovie);
begin
  FlatCommaText(List, Movie.strProducer, Movie);
end;

procedure FlatCountry(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strCountry, Movie);
end;

procedure FlatLanguage(List: TStringList; Movie: TMovie);
begin
  FlatCommaText(List, Movie.strLanguages, Movie);
end;

procedure FlatNumber(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iNumber), Movie);
end;

procedure FlatMedia(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strMedia, Movie);
end;

procedure FlatMediaType(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strMediaType, Movie);
end;

procedure FlatDate(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iDate), Movie);
end;

procedure FlatBorrower(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strBorrower, Movie);
end;

procedure FlatLength(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iLength), Movie);
end;

procedure FlatActors(List: TStringList; Movie: TMovie);
begin
  FlatCommaText(List, Movie.strActors, Movie);
end;

procedure FlatVideoFormat(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strVideoFormat, Movie);
end;

procedure FlatVideoBitrate(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iVideoBitrate), Movie);
end;

procedure FlatAudioFormat(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strAudioFormat, Movie);
end;

procedure FlatAudioBitrate(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iAudioBitrate), Movie);
end;

procedure FlatResolution(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strResolution, Movie);
end;

procedure FlatFramerate(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strFramerate, Movie);
end;

procedure FlatSubtitles(List: TStringList; Movie: TMovie);
begin
  FlatCommaText(List, Movie.strSubtitles, Movie);
end;

procedure FlatDisks(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iDisks), Movie);
end;

//------------------------------------------------------------------------------

procedure AddOnceCategory(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strCategory, Movie);
end;

procedure AddOnceOriginalTitle(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strOriginalTitle, Movie);
end;

procedure AddOnceTranslatedTitle(List: TStringList; Movie: TMovie);
begin
  if (Movie.strTranslatedTitle <> '')
    then List.AddObject(Movie.strTranslatedTitle, Movie)
    else List.AddObject(Movie.strOriginalTitle, Movie);
end;

procedure AddOnceRating(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iRating), Movie);
end;

procedure AddOnceYear(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iYear), Movie);
end;

procedure AddOnceDecade(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iYear), Movie);
end;

procedure AddOnceDirector(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strDirector, Movie);
end;

procedure AddOnceProducer(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strProducer, Movie);
end;

procedure AddOnceCountry(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strCountry, Movie);
end;

procedure AddOnceLanguage(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strLanguages, Movie);
end;

procedure AddOnceNumber(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iNumber), Movie);
end;

procedure AddOnceMedia(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strMedia, Movie);
end;

procedure AddOnceMediaType(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strMediaType, Movie);
end;

procedure AddOnceDate(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iDate), Movie);
end;

procedure AddOnceBorrower(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strBorrower, Movie);
end;

procedure AddOnceLength(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iLength), Movie);
end;

procedure AddOnceActors(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strActors, Movie);
end;

procedure AddOnceVideoFormat(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strVideoFormat, Movie);
end;

procedure AddOnceVideoBitrate(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iVideoBitrate), Movie);
end;

procedure AddOnceAudioFormat(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strAudioFormat, Movie);
end;

procedure AddOnceAudioBitrate(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iAudioBitrate), Movie);
end;

procedure AddOnceResolution(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strResolution, Movie);
end;

procedure AddOnceFramerate(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strFramerate, Movie);
end;

procedure AddOnceSubtitles(List: TStringList; Movie: TMovie);
begin
  List.AddObject(Movie.strSubtitles, Movie);
end;

procedure AddOnceDisks(List: TStringList; Movie: TMovie);
begin
  List.AddObject(IntToStr(Movie.iDisks), Movie);
end;

//------------------------------------------------------------------------------


procedure GroupString(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
begin
  SameGroup := SameText(Item1, Item2);
  if not SameGroup then
  begin
    if (Item2 <> '')
      then NewGroup := Item2
      else NewGroup := _('< n/a >');
  end;
end;

procedure GroupAlphabet(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
begin
  SameGroup := Item1 = Item2;
  if not SameGroup then
  begin
    if (Item2 <> '')
      then NewGroup := _('Letter') + ' ' + Item2
      else NewGroup := _('< n/a >');
  end;
end;

procedure GroupPeople(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
begin
  SameGroup := UpperCase(Copy(Item1, 1, 1)) = UpperCase(Copy(Item2, 1, 1));
  if not SameGroup then
  begin
    if (Item2 <> '')
      then NewGroup := _('Letter') + ' ' + UpperCase(Copy(Item2, 1, 1))
      else NewGroup := _('< n/a >');
  end;
end;

procedure GroupRating(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
var
  i1, i2: Integer;
begin
  i1 := StrToIntDef(Item1, 0) div 10;
  i2 := StrtoIntDef(Item2, 0) div 10;
  SameGroup := i1 = i2;
  if not SameGroup then
  begin
    if (i2 <= 0)
      then NewGroup := _('< n/a >')
      else NewGroup := _('Rating') + ' ' + IntToStr(i2);
  end;
end;

procedure GroupYear(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
var
  i1, i2: Integer;
begin
  i1 := StrToIntDef(Item1, 0);
  i2 := StrtoIntDef(Item2, 0);
  SameGroup := i1 = i2;
  if not SameGroup then
  begin
    if (i2 <= 0)
      then NewGroup := _('< n/a >')
      else NewGroup := _('Year') + ' ' + IntToStr(i2);
  end;
end;

procedure GroupDecade(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
var
  i1, i2: Integer;
begin
  i1 := StrToIntDef(Item1, 0) div 10;
  i2 := StrtoIntDef(Item2, 0) div 10;
  SameGroup := i1 = i2;
  if not SameGroup then
  begin
    if (i2 <= 0)
      then NewGroup := _('< n/a >')
      else NewGroup := _('Decade') + ' ' + IntToStr(i2 * 10);
  end;
end;

procedure GroupNumber(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
var
  i1, i2: Integer;
begin
  i1 := StrToIntDef(Item1, 0) div 50;
  i2 := StrtoIntDef(Item2, 0) div 50;
  SameGroup := i1 = i2;
  if not SameGroup
    then NewGroup := _('Number') + ' ' + IntToStr(i2 * 50) + ' - ' + IntToStr((i2 * 50) + 49);
end;

procedure GroupDate(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
var
  i1, i2: Integer;
begin
  i1 := YearOf(StrToIntDef(Item1, 0));
  i2 := YearOf(StrtoIntDef(Item2, 0));
  SameGroup := i1 = i2;
  if not SameGroup
    then NewGroup := _('Year') + ' ' + IntToStr(i2);
end;

procedure GroupLength(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
var
  i1, i2: Integer;
begin
  i1 := StrToIntDef(Item1, 0) div 10;
  i2 := StrtoIntDef(Item2, 0) div 10;
  SameGroup := i1 = i2;
  if not SameGroup then
  begin
    if (i2 <= 0)
      then NewGroup := _('< n/a >')
      else NewGroup := IntToStr(i2 * 10) + ' ' + _('Minutes');
  end;
end;

procedure GroupVideoBitrate(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
var
  i1, i2: Integer;
begin
  i1 := StrToIntDef(Item1, 0);
  i2 := StrtoIntDef(Item2, 0);
  SameGroup := i1 = i2;
  if not SameGroup then
  begin
    if (i2 <= 0)
      then NewGroup := _('< n/a >')
      else NewGroup := IntToStr(i2) + ' ' + _('Kbps');
  end;
end;

procedure GroupAudioBitrate(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
var
  i1, i2: Integer;
begin
  i1 := StrToIntDef(Item1, 0);
  i2 := StrtoIntDef(Item2, 0);
  SameGroup := i1 = i2;
  if not SameGroup then
  begin
    if (i2 <= 0)
      then NewGroup := _('< n/a >')
      else NewGroup := IntToStr(i2) + ' ' + _('Kbps');
  end;
end;

procedure GroupDisks(const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);
var
  i1, i2: Integer;
begin
  i1 := StrToIntDef(Item1, 0);
  i2 := StrtoIntDef(Item2, 0);
  SameGroup := i1 = i2;
  if not SameGroup then
  begin
    if (i2 <= 0)
      then NewGroup := _('< n/a >') else
    if (i2 = 1)
      then NewGroup := _('1 Disc')
      else NewGroup := IntToStr(i2) + ' ' + _('Discs');
  end;
end;


//------------------------------------------------------------------------------


type
  TFlatProc = procedure (List: TStringList; Movie: TMovie);
  TGroupProc = procedure (const Item1, Item2: string; var SameGroup: Boolean; var NewGroup: string);

  TGroupRec = record
    GroupAddProc: TFlatProc;
    GroupSortProc: TStringListSortCompare;
    GroupProc: TGroupProc;
    OrderAddProc: TFlatproc;
    OrderSortProc: TStringListSortCompare;
    Title: string;
  end;

const
  FlatProcTables: array [TSortKind] of TFlatProc = (
    nil, FlatCategory, FlatOriginalTitle, FlatTranslatedTitle, FlatRating, FlatYear, FlatDecade,
    FlatDirector, FlatProducer, FlatCountry, FlatLanguage, FlatNumber,
    FlatMedia, FlatMediaType, FlatDate, FlatBorrower, FlatLength, FlatActors,
    FlatVideoFormat, FlatVideoBitrate, FlatAudioFormat, FlatAudioBitrate,
    FlatResolution, FlatFramerate, FlatSubtitles, FlatDisks);

  AddOnceProcTables: array [TSortKind] of TFlatProc = (
    nil, AddOnceCategory, AddOnceOriginalTitle, AddOnceTranslatedTitle, AddOnceRating, AddOnceYear, AddOnceDecade,
    AddOnceDirector, AddOnceProducer, AddOnceCountry, AddOnceLanguage, AddOnceNumber,
    AddOnceMedia, AddOnceMediaType, AddOnceDate, AddOnceBorrower, AddOnceLength, AddOnceActors,
    AddOnceVideoFormat, AddOnceVideoBitrate, AddOnceAudioFormat, AddOnceAudioBitrate,
    AddOnceResolution, AddOnceFramerate, AddOnceSubtitles, AddOnceDisks);

  SortAscProcTables: array [TSortKind] of TStringListSortCompare = (
    nil, SortStrAscd, SortStrAscd, SortStrAscd, SortIntAscd, SortIntAscd, SortIntAscd,
    SortStrAscd, SortStrAscd, SortStrAscd, SortStrAscd, SortIntAscd,
    SortStrAscd, SortStrAscd, SortIntAscd, SortStrAscd, SortIntAscd, SortStrAscd,
    SortStrAscd, SortIntAscd, SortStrAscd, SortIntAscd,
    SortStrAscd, SortStrAscd, SortStrAscd, SortIntAscd);

  SortDscProcTables: array [TSortKind] of TStringListSortCompare = (
    nil, SortStrDscd, SortStrDscd, SortStrDscd, SortIntDscd, SortIntDscd, SortIntDscd,
    SortStrDscd, SortStrDscd, SortStrDscd, SortStrDscd, SortIntDscd,
    SortStrDscd, SortStrDscd, SortIntDscd, SortStrDscd, SortIntDscd, SortStrDscd,
    SortStrDscd, SortIntDscd, SortStrDscd, SortIntDscd,
    SortStrDscd, SortStrDscd, SortStrDscd, SortIntDscd);

  GroupProcTables: array [TSortKind] of TGroupProc = (
    nil, GroupString, GroupAlphabet, GroupAlphabet, GroupRating, GroupYear, GroupDecade,
    GroupPeople, GroupPeople, GroupString, GroupString, GroupNumber,
    GroupString, GroupString, GroupDate, GroupString, GroupLength, GroupPeople,
    GroupString, GroupVideoBitrate, GroupString, GroupAudioBitrate,
    GroupString, GroupString, GroupString, GroupDisks);

  GroupTitleTables: array [TSortKind] of string = (
    '', 'Category', 'Original Title', 'Translated Title', 'Rating', 'Year', 'Decade',
    'Director', 'Producer', 'Country', 'Language', 'Number',
    'Media', 'Media Type', 'Date', 'Borrower', 'Length', 'Actors',
    'Video Format', 'Video Bitrate', 'Audio Format', 'Audio Bitrate',
    'Resolution', 'Framerate', 'Subtitles', 'Disks');

procedure GetGroupRec(GroupKind: TSortKind; GroupReverseOrder: Boolean;
  SortKind: TSortKind; SortReverseOrder: Boolean; var Info: TGroupRec);
var
  sk: TSortKind;
begin
  Info.GroupAddProc := FlatProcTables[GroupKind];
  if GroupReverseOrder
    then Info.GroupSortProc := SortDscProcTables[GroupKind]
    else Info.GroupSortProc := SortAscProcTables[GroupKind];
  Info.GroupProc := GroupProcTables[GroupKind];
  if (SortKind = skNone)
    then sk := DefaultSortKind
    else sk := SortKind;
  Info.OrderAddProc := AddOnceProcTables[sk];
  if SortReverseOrder
    then Info.OrderSortProc := SortDscProcTables[sk]
    else Info.OrderSortProc := SortAscProcTables[sk];
  Info.Title := '';
  if (GroupKind <> skNone)
    then Info.Title := _('Group by') + ' ' + _(GroupTitleTables[GroupKind]) + ', ';
  Info.Title := Info.Title + _('Order by') + ' ' + _(GroupTitleTables[sk]);
end;

procedure GroupAndSortMovies(List: TList; Bar: TNiceSideBar; TitleType: Integer;
  GroupKind: TSortKind; GroupReverseOrder: Boolean;
  SortKind: TSortKind; SortReverseOrder: Boolean;
  var Title: string);
label
  Done;
var
  Info: TGroupRec;
  t: TStringList;
  x, y: Integer;
  Section: TSideBarItem;
  SameGroup: Boolean;
  NewGroup: string;
  TitleProc: TFlatProc;
  IsPeople: Boolean;
  CurrMovies, CurrPerson: string;
  m: TStringList;
begin
  Bar.BeginUpdate;
  t := TStringList.Create;
  try
    if (Bar.Tag = 1) then
    begin
      for x := 0 to Bar.Items.Count-1 do
      begin
        m := TStringList(Bar.Items[x].Tag);
        if Assigned(m)
          then m.Free;
      end;
    end;  
    Bar.Items.Clear;
    Bar.Tag := 0;
    if (List.Count = 0)
      then goto Done;
    GetGroupRec(GroupKind, GroupReverseOrder, SortKind, SortReverseOrder, Info);
    Title := Info.Title;
    if Assigned(Info.GroupAddProc) then
    begin
      for x := 0 to List.Count-1
        do Info.GroupAddProc(t, TMovie(List[x]));
      if Assigned(Info.GroupSortProc)
        then t.CustomSort(Info.GroupSortProc)
        else t.Sort;
      Info.GroupProc('$FEFEFEFE', t[0], SameGroup, NewGroup);
      Section := Bar.Items.Add;
      Section.Caption := NewGroup;
      Section.Expanded := False;
      Section.Items.AddObject(t[0], t.Objects[0]);
      for x := 1 to t.Count-1 do
      begin
        Info.GroupProc(t[x-1], t[x], SameGroup, NewGroup);
        if not SameGroup then
        begin
          Section := Bar.Items.Add;
          Section.Caption := NewGroup;
          Section.Expanded := False;
        end;
        Section.Items.AddObject(t[x], t.Objects[x]);
      end;
    end else
    begin
      Section := Bar.Items.Add;
      Section.Caption := _('All Movies');
      for x := 0 to List.Count-1
        do Section.Items.AddObject('', TObject(List[x]));
    end;
    IsPeople := @Info.GroupProc = @GroupPeople;
    if IsPeople then
    begin
      Bar.Tag := 1;
      for x := 0 to Bar.Items.Count-1 do
      begin
        Section := Bar.Items[x];
        if (Section.Items.Count = 0)
          then Continue;
        t.Clear;
        t.Assign(Section.Items);
        if SortReverseOrder
          then t.CustomSort(SortStrDscd)
          else t.Sort;
        Section.Items.Clear;
        m := TStringList.Create;
        Section.Tag := Integer(m);
        CurrPerson := t[0];
        CurrMovies := IntToStr(Integer(t.Objects[0]));
        for y := 1 to t.Count-1 do
        begin
          if (t[y] <> CurrPerson) then
          begin
            Section.Items.Add(CurrPerson);
            m.Add(CurrMovies);
            CurrPerson := t[y];
            CurrMovies := '';
          end;
          if (CurrMovies <> '')
            then CurrMovies := CurrMovies + ',';
          CurrMovies := CurrMovies + IntToStr(Integer(t.Objects[y]));
        end;
        if (CurrMovies <> '') then
        begin
          Section.Items.Add(CurrPerson);
          m.Add(CurrMovies);
        end;

        Section.Caption := Section.Caption + ' (' + IntToStr(Section.Items.Count) + ')';

      end;

    end else
    begin
      if (TitleType = 1)
        then TitleProc := AddOnceTranslatedTitle
        else TitleProc := AddOnceOriginalTitle;
      for x := 0 to Bar.Items.Count-1 do
      begin
        Section := Bar.Items[x];
        Section.Caption := Section.Caption + ' (' + IntToStr(Section.Items.Count) + ')';
        t.Clear;
        for y := 0 to Section.Items.Count-1
          do Info.OrderAddProc(t, TMovie(Section.Items.Objects[y]));
        if Assigned(Info.OrderSortProc)
          then t.CustomSort(Info.OrderSortProc)
          else t.Sort;
        Section.Items.Clear;
        for y := 0 to t.Count-1
          do TitleProc(Section.Items, TMovie(t.Objects[y]));
      end;
    end;
    Done:
  finally
    Bar.EndUpdate;
    t.Free;
  end;
end;

end.

